<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// --- LISTE CORRECTE BASÉE SUR VOS FICHIERS ---
use App\Models\Analgesies;
use App\Models\AntecedentsTerrain;
use App\Models\DetailsTherapeutiquesImplants;
use App\Models\DifficultesOperatoires;
use App\Models\DonneesAnesthesique;
use App\Models\DonneesBiologique;
use App\Models\DonneesClinique;
use App\Models\GreffesOsseuses;
use App\Models\ImageriesDiagnostic;
use App\Models\Implant;
use App\Models\Intervention;
use App\Models\Patient;
use App\Models\ProphylaxiesAntibiotiques;
use App\Models\Revision;
use App\Models\SuitesOperatoire;
use App\Models\Surgeon;
use App\Models\SystemesNavigation;
use Illuminate\Support\Facades\Validator;

class FormulaireController extends Controller
{
    /**
     * Affiche une liste paginée des interventions POUR UN CHIRURGIEN SPÉCIFIQUE.
     * C'est la méthode READ (pour une liste).
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        // 1. VALIDATION : L'UID du chirurgien est obligatoire.
        $validated = $request->validate([
            'surgeon_firebase_uid' => 'required|string'
        ]);

        // 2. CRÉATION OU RÉCUPÉRATION DU CHIRURGIEN (LOGIQUE CONSERVÉE)
        // C'est un excellent pattern pour votre application.
        $surgeon = $request->attributes->get('surgeon');

        // 3. REQUÊTE SCOPÉE ET OPTIMISÉE
        $interventions = Intervention::query()
            // SÉCURITÉ : On ne prend que les interventions de ce chirurgien.
            ->where('surgeon_id', $surgeon->id)

            // OPTIMISATION : On charge uniquement les colonnes utiles du patient pour la liste.
            // Cela réduit la taille de la réponse et améliore la vitesse.
            ->with('patient:id,nom,prenom,date_naissance') // On ajoute la date de naissance, souvent utile dans les listes.

            // ERGONOMIE : On trie par la date d'intervention la plus récente.
            ->latest('date_intervention')

            // PERFORMANCE : On pagine les résultats.
            ->paginate(15); // Vous pouvez ajuster le nombre

        // 4. RÉPONSE JSON
        return response()->json($interventions);
    }

    /**
     * Enregistre un nouveau formulaire complet dans la base de données.
     * VERSION COMPLÈTE ET CORRIGÉE
     */
    public function store(Request $request)
    {
        try {
            // Validation des champs obligatoires avec messages personnalisés
            $validator = Validator::make($request->all(), [
                'surgeon_firebase_uid' => 'required|string',
                'formType' => 'required|string|in:hanche,genou',
                'IDENTITE DU PATIENT' => 'required|array',
                'IDENTITE DU PATIENT.Nom' => 'required|string|max:255',
                'DONNEES CLINIQUES' => 'required|array',
                'Anesthésie' => 'required|array',
            ], [
                'surgeon_firebase_uid.required' => 'L\'identifiant du chirurgien est obligatoire.',
                'formType.required' => 'Le type de formulaire (hanche/genou) est obligatoire.',
                'formType.in' => 'Le type de formulaire doit être "hanche" ou "genou".',
                'IDENTITE DU PATIENT.required' => 'Les informations du patient sont obligatoires.',
                'IDENTITE DU PATIENT.Nom.required' => 'Le nom du patient est obligatoire.',
                'DONNEES CLINIQUES.required' => 'Les données cliniques sont obligatoires.',
                'Anesthésie.required' => 'Les données d\'anesthésie sont obligatoires.',
            ]);

            // Si la validation échoue, retourner des erreurs claires
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erreur de validation',
                    'errors' => $validator->errors(),
                    'failed_fields' => array_keys($validator->errors()->messages())
                ], 422);
            }

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Données invalides',
                'error' => 'Erreur de validation : ' . $e->getMessage(),
                'details' => $e->errors()
            ], 422);
        }

        // La transaction garantit la cohérence des données
        DB::beginTransaction();

        try {
            // == 1. Création ou récupération du chirurgien ==
            try {
                // On récupère le chirurgien que le middleware a déjà trouvé pour nous.
                $surgeon = $request->attributes->get('surgeon');
                Log::info('Chirurgien trouvé/créé', ['surgeon_id' => $surgeon->id]);
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création/récupération du chirurgien : ' . $e->getMessage());
            }

            // == 2. Création ou récupération du patient ==
            try {
                $patientData = $request->input('IDENTITE DU PATIENT');

                if (empty($patientData['Nom'])) {
                    throw new \Exception('Le nom du patient est manquant.');
                }

                // Extraction correcte de la profession (format: {value: "...", autre: "..."})
                $profession = null;
                $professionAutre = null;
                if (isset($patientData['Profession'])) {
                    if (is_array($patientData['Profession'])) {
                        $profession = $patientData['Profession']['value'] ?? null;
                        $professionAutre = $patientData['Profession']['autre'] ?? null;
                    } else {
                        $profession = $patientData['Profession'];
                    }
                }

                $criteresUnicite = [];
                if (!empty($patientData['Numéro Identifiant'])) {
                    $criteresUnicite['numero_identifiant'] = $patientData['Numéro Identifiant'];
                } else {
                    $criteresUnicite = [
                        'nom' => $patientData['Nom'],
                        'prenom' => $patientData['Prénom(s)'] ?? null,
                        'date_naissance' => $this->parseDate($patientData['Date de naissance (JJ-MM-AAAA)'] ?? null),
                    ];
                }

                $donneesPatient = [
                    'nom' => $patientData['Nom'],
                    'prenom' => $patientData['Prénom(s)'] ?? null,
                    'date_naissance' => $this->parseDate($patientData['Date de naissance (JJ-MM-AAAA)'] ?? null),
                    'sexe' => $patientData['Sexe'] ?? null,
                    'profession' => $profession,
                    'profession_autre' => $professionAutre,
                    'adresse' => $patientData['Adresse du patient'] ?? null,
                    'telephone' => $patientData['Téléphone'] ?? null,
                    'numero_identifiant' => $patientData['Numéro Identifiant'] ?? null,
                ];

                $patient = Patient::updateOrCreate($criteresUnicite, $donneesPatient);
                Log::info('Patient créé/mis à jour', ['patient_id' => $patient->id]);
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la gestion du patient : ' . $e->getMessage());
            }

            // == 3. Création de l'intervention ==
            try {
                $interventionData = $request->input('DONNEES THERAPEUTIQUES.Chirurgicales', []);
                $chirurgienData = $request->input('Chirurgien', []);
                $formType = $request->input('formType');

                // Gestion voie d'abord selon le type
                $voieAbord = null;
                $voieAbordAutre = null;

                if ($formType === 'hanche') {
                    $voieAbordData = $interventionData["Voie d'abord de la hanche"] ?? null;
                    if (is_array($voieAbordData)) {
                        $voieAbord = $voieAbordData['value'] ?? null;
                        $voieAbordAutre = $voieAbordData['autre'] ?? null;
                    } else {
                        $voieAbord = $voieAbordData;
                    }
                } else {
                    // Genou: voie d'abord est un tableau de checkboxes
                    $voieAbordArray = $interventionData["Voie d'abord"] ?? [];
                    $voieAbord = is_array($voieAbordArray) ? json_encode($voieAbordArray) : $voieAbordArray;
                }

                // Extraction hôpital
                $hopitalData = $patientData["Hôpital / numéro d'établissement"] ?? null;
                $hopital = null;
                if (is_array($hopitalData)) {
                    $hopital = $hopitalData['value'] ?? $hopitalData['autre'] ?? null;
                    if ($hopital === 'autre') {
                        $hopital = $hopitalData['autre'] ?? null;
                    }
                } else {
                    $hopital = $hopitalData;
                }

                // Création de l'intervention
                $intervention = $patient->interventions()->create([
                    'surgeon_id' => $surgeon->id,
                    'type_prothese' => $formType,
                    'hopital_etablissement' => $hopital,
                    'date_intervention' => $this->parseDate($interventionData['Date intervention'] ?? null),
                    'heure_debut' => $interventionData['Heure début'] ?? null,
                    'heure_fin' => $interventionData['Fin'] ?? null,
                    'chirurgien_identifiant' => $chirurgienData['Identifiant/code'] ?? null,
                    'voie_dabord' => $voieAbord,
                    'voie_dabord_autre' => $voieAbordAutre,
                    'donnees_formulaire_autres' => $this->extractAutresDonnees($request, $formType),
                ]);
                Log::info('Intervention créée', ['intervention_id' => $intervention->id]);
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création de l\'intervention : ' . $e->getMessage());
            }

            // == 4. Antécédents ==
            try {
                $antecedentsData = $request->input('TERRAIN (S) - ANTECEDENTS') ?? $request->input('TERRAIN (S) - ANTECEDANTS');
                if ($antecedentsData) {
                    // Extraction intervention chirurgicale antérieure
                    $interventionAnt = null;
                    $interventionAntAutre = null;

                    $interventionAntData = $antecedentsData['Intervention chirurgicale sur cette hanche']
                        ?? $antecedentsData['Intervention chirurgicale sur ce genou']
                        ?? null;

                    if (is_array($interventionAntData)) {
                        $interventionAnt = $interventionAntData['value'] ?? null;
                        $interventionAntAutre = $interventionAntData['autre'] ?? null;
                    } else {
                        $interventionAnt = $interventionAntData;
                    }

                    $intervention->antecedentsTerrains()->create([
                        'antecedents_medicaux' => $antecedentsData['Antécédents médicaux'] ?? [],
                        'serologie_retroviral' => $antecedentsData['Sérologie rétroviral'] ?? null,
                        'traumatisme' => $this->parseBoolean($antecedentsData['Traumatisme'] ?? null),
                        'corticotherapie' => $this->parseBoolean($antecedentsData['Corticothérapie'] ?? null),
                        'intervention_anterieure' => $interventionAnt,
                        'intervention_anterieure_autre' => $interventionAntAutre,
                    ]);
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des antécédents : ' . $e->getMessage());
            }

            // == 5. DONNÉES CLINIQUES ==
            try {
                $cliniquesData = $request->input('DONNEES CLINIQUES');

                $poids = $this->parseNumeric($cliniquesData['Poids (kg)'] ?? null);
                $taille_cm = $this->parseNumeric($cliniquesData['Taille (cm)'] ?? null);
                $imc = $this->parseNumeric($cliniquesData['IMC (kg/m²)'] ?? null);

                if (is_null($imc) && $poids && $taille_cm && $taille_cm > 0) {
                    $taille_m = $taille_cm / 100;
                    $imc = round($poids / ($taille_m * $taille_m), 2);
                }

                // Extraction impotence fonctionnelle
                $impotence = false;
                $impotencePrecision = null;
                $impotenceData = $cliniquesData['Impotence fonctionnelle'] ?? null;

                if (is_array($impotenceData)) {
                    $impotence = $this->parseBoolean($impotenceData['value'] ?? null);
                    $impotencePrecision = $impotenceData['conditional_value'] ?? null;
                } else {
                    $impotence = $this->parseBoolean($impotenceData);
                }

                // Extraction ILMI
                $ilmi = false;
                $ilmiCm = null;
                $ilmiData = $cliniquesData['ILMI'] ?? null;

                if (is_array($ilmiData)) {
                    $ilmi = $this->parseBoolean($ilmiData['value'] ?? null);
                    $ilmiCm = $this->parseNumeric($ilmiData['autre'] ?? null);
                } else {
                    $ilmi = $this->parseBoolean($ilmiData);
                }

                $intervention->donneesCliniques()->create([
                    'poids_kg' => $poids,
                    'taille_cm' => $taille_cm,
                    'imc' => $imc,
                    'cote_opere' => $cliniquesData['Côté'] ?? null,
                    'boiterie' => $this->parseBoolean($cliniquesData['Boiterie'] ?? null),
                    'impotence_fonctionnelle' => $impotence,
                    'impotence_fonctionnelle_precision' => $impotencePrecision,
                    'pma_pre_operatoire' => $cliniquesData['PMA'] ?? null,
                    'score_parker' => $cliniquesData['Score de Parker'] ?? null,
                    'ilmi' => $ilmi,
                    'ilmi_cm' => $ilmiCm,
                ]);
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données cliniques : ' . $e->getMessage());
            }

            // == 6. DONNÉES PARACLINIQUES ==
            try {
                $paracliniquesData = $request->input('DONNEES PARACLINIQUES');

                if ($paracliniquesData) {
                    // Biologie
                    if (isset($paracliniquesData['Biologie'])) {
                        $bioData = $paracliniquesData['Biologie'];
                        $intervention->donneesBiologiques()->create([
                            'globules_blancs' => $this->parseNumeric($bioData['Globules Blancs (10³/mm³)'] ?? null),
                            'neutrophiles' => $this->parseNumeric($bioData['Neutrophiles (10³/mm³)'] ?? $bioData['Neutrophiles'] ?? null),
                            'hemoglobine' => $this->parseNumeric($bioData['Hémoglobine (g/dl)'] ?? null),
                            'plaquettes' => $this->parseNumeric($bioData['Plaquettes (10³/mm³)'] ?? null),
                            'groupe_sanguin' => $bioData['Groupe Sanguin'] ?? null,
                            'rhesus' => $bioData['Rhésus'] ?? null,
                            'creatinine' => $this->parseNumeric($bioData['Créatininémie (µmol/L)'] ?? $bioData['Créatinine (mmol/L)'] ?? null),
                            'glycemie' => $this->parseNumeric($bioData['Glycémie (mmol/L)'] ?? null),
                            'crp' => $this->parseNumeric($bioData['CRP (mg/L)'] ?? $bioData['CRP (g/L)'] ?? null),
                            'electrophorese_hemoglobine' => $bioData['Electrophorèse de l\'hémoglobine'] ?? null,
                            'procalcitonine' => $bioData['Procalcitonine (ng/mL)'] ?? $bioData['Procalcitonine'] ?? null,
                            'tp_tck' => $bioData['TP/TCK'] ?? null,
                            'inr' => $bioData['INR'] ?? null,
                            'autre_biologie' => $bioData['Autre (préciser)'] ?? null,
                        ]);
                    }
                    // Imagerie
                    if (isset($paracliniquesData['Imagerie (Diagnostic)'])) {
                        $imagerieData = $paracliniquesData['Imagerie (Diagnostic)'];

                        $resultatPrincipal = null;
                        $resultatPrincipalAutre = null;

                        $diagnosticData = $imagerieData['Résultats (cocher le principal)']
                            ?? $imagerieData['Diagnostic (cocher le principal)']
                            ?? null;

                        if (is_array($diagnosticData)) {
                            $resultatPrincipal = $diagnosticData['value'] ?? null;
                            $resultatPrincipalAutre = $diagnosticData['autre'] ?? null;
                        } else {
                            $resultatPrincipal = $diagnosticData;
                        }

                        $intervention->imageriesDiagnostics()->create([
                            'techniques' => json_encode($imagerieData['Techniques'] ?? []),
                            'resultat_principal' => $resultatPrincipal,
                            'resultat_principal_autre' => $resultatPrincipalAutre,
                        ]);
                    }
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données paracliniques : ' . $e->getMessage());
            }

            // == 7. ANESTHÉSIE ==
            try {
                $anesthesieData = $request->input('Anesthésie');

                $typeAnesthesie = null;
                $typeAnesthesieAutre = null;
                $typeData = $anesthesieData['Type'] ?? null;

                if (is_array($typeData)) {
                    $typeAnesthesie = $typeData['value'] ?? null;
                    $typeAnesthesieAutre = $typeData['autre'] ?? null;
                } else {
                    $typeAnesthesie = $typeData;
                }

                $complicationsPerAnesth = false;
                $complicationsPerAnesthDetails = null;
                $complicationsData = $anesthesieData['Complications per-anesthésiques'] ?? null;

                if (is_array($complicationsData)) {
                    $complicationsPerAnesth = $this->parseBoolean($complicationsData['value'] ?? null);
                    $complicationsPerAnesthDetails = $complicationsData['autre'] ?? null;
                } else {
                    $complicationsPerAnesth = $this->parseBoolean($complicationsData);
                }

                $intervention->donneesAnesthesiques()->create([
                    'type_anesthesie' => $typeAnesthesie,
                    'type_anesthesie_autre' => $typeAnesthesieAutre,
                    'classification_asa' => $anesthesieData['Classification ASA'] ?? null,
                    'complications_per_anesthesiques' => $complicationsPerAnesth,
                    'complications_per_anesthesiques_details' => $complicationsPerAnesthDetails,
                ]);
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données d\'anesthésie : ' . $e->getMessage());
            }

            // == 8. SUITES OPÉRATOIRES ==
            try {
                $suitesData = $request->input('Suites opératoires et Résultats');

                if ($suitesData) {
                    // Prise en charge des frais
                    $priseEnCharge = null;
                    $priseEnChargeAutre = null;
                    $priseEnChargeData = $suitesData['Prise en charge des frais de soins'] ?? null;

                    if (is_array($priseEnChargeData)) {
                        $priseEnCharge = $priseEnChargeData['value'] ?? null;
                        $priseEnChargeAutre = $priseEnChargeData['autre'] ?? null;
                    } else {
                        $priseEnCharge = $priseEnChargeData;
                    }

                    // Rééducation fonctionnelle
                    $reeducation = false;
                    $reeducationNbSeances = null;
                    $reeducationData = $suitesData['Rééducation fonctionnelle'] ?? null;

                    if (is_array($reeducationData)) {
                        $reeducation = $this->parseBoolean($reeducationData['value'] ?? null);
                        $reeducationNbSeances = $reeducationData['autre'] ?? null;
                    } else {
                        $reeducation = $this->parseBoolean($reeducationData);
                    }

                    // Correction d'un raccourcissement
                    $correctionRaccourcissement = false;
                    $correctionRaccourcissementCm = null;
                    $correctionRaccData = $suitesData['Correction d\'un raccourcissement'] ?? null;

                    if (is_array($correctionRaccData)) {
                        $correctionRaccourcissement = $this->parseBoolean($correctionRaccData['value'] ?? null);
                        $correctionRaccourcissementCm = $this->parseNumeric($correctionRaccData['autre'] ?? null);
                    } else {
                        $correctionRaccourcissement = $this->parseBoolean($correctionRaccData);
                    }

                    // Correction d'une boiterie
                    $correctionBoiterie = $this->parseBoolean($suitesData['Correction d\'une boiterie'] ?? null);

                    // Si descellement
                    $descellementType = null;
                    $descellementSiege = null;
                    $siDescellement = $suitesData['Si descellement'] ?? null;

                    if (is_array($siDescellement)) {
                        $descellementType = $siDescellement['Type'] ?? null;
                        $descellementSiege = $siDescellement['Siège'] ?? null;
                    }

                    $intervention->suitesOperatoires()->create([
                        'delai_mise_en_charge_jours' => $this->parseNumeric($suitesData['Délai de mise en charge (jours)'] ?? null),
                        'delai_appui_total_mois' => $this->parseNumeric($suitesData['Délai d\'appui total (mois)'] ?? null),
                        'duree_sejour_hospitalier' => $suitesData['Durée de séjour hospitalier'] ?? null,
                        'reeducation_fonctionnelle' => $reeducation,
                        'reeducation_nb_seances' => $reeducationNbSeances,
                        'complications_per_operatoire' => json_encode($suitesData['Complications per opératoire'] ?? []),
                        'suites_immediates' => $suitesData['Suites opératoires immédiates'] ?? null,
                        'complications_immediates' => json_encode($suitesData['Complications immédiates'] ?? []),
                        'complications_secondaires' => json_encode($suitesData['Complications secondaires'] ?? []),
                        'complications_tardives' => json_encode($suitesData['Complications tardives'] ?? []),
                        'prise_en_charge_frais' => $priseEnCharge,
                        'prise_en_charge_frais_autre' => $priseEnChargeAutre,
                        'dernier_recul_mois' => $suitesData['Dernier recul (mois)'] ?? null,
                        'eva_patient' => $suitesData['Appréciation du patient lui-même : EVA'] ?? null,
                        'correction_raccourcissement' => $correctionRaccourcissement,
                        'correction_raccourcissement_cm' => $correctionRaccourcissementCm,
                        'correction_boiterie' => $correctionBoiterie,
                        'pma_post_operatoire' => $suitesData['Evaluation fonctionnelle : PMA'] ?? null,
                        'descellement_type' => $descellementType,
                        'descellement_siege' => $descellementSiege,
                    ]);
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des suites opératoires : ' . $e->getMessage());
            }

            // == 9. RÉVISION ==
            try {
                $revisionData = $request->input('En cas de révision');

                if ($revisionData) {
                    $typesRevisionData = $revisionData['Type de révision'] ?? [];
                    $motifsRevisionData = $revisionData['Motif de la révision'] ?? [];

                    $typesRevision = $typesRevisionData['Type'] ?? [];
                    $typesRevisionAutre = null;

                    // Vérifier si "Autre" est dans les types et extraire la valeur
                    if (is_array($typesRevision) && in_array('Autre', $typesRevision)) {
                        $typesRevisionAutre = $typesRevisionData['Lequel(s) / Autre (préciser)'] ?? null;
                    }

                    $motifsRevision = $motifsRevisionData['Motif'] ?? [];
                    $motifsRevisionAutre = null;

                    // Vérifier si "Autre" est dans les motifs
                    if (is_array($motifsRevision) && in_array('Autre', $motifsRevision)) {
                        $motifsRevisionAutre = $motifsRevisionData['Fracture Spécifiée / Autre'] ?? null;
                    }

                    if (!empty($typesRevision) || !empty($motifsRevision)) {
                        $intervention->revision()->create([
                            'est_une_revision' => true,
                            'types_revision' => json_encode($typesRevision),
                            'types_revision_autre' => $typesRevisionAutre,
                            'motifs_revision' => json_encode($motifsRevision),
                            'motifs_revision_autre' => $motifsRevisionAutre,
                        ]);
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Erreur non bloquante - révision : ' . $e->getMessage());
            }

            // == 10. IMPLANTS ==
            $this->createImplants($intervention, $request, $formType);

            // == 11. DÉTAILS THÉRAPEUTIQUES ==
            $this->createDetailsTherapeutiques($intervention, $request, $formType);

            // == 12. GREFFE OSSEUSE ==
            try {
                $greffeData = $request->input('Greffe osseuse');

                if ($greffeData && $this->parseBoolean($greffeData['Transplantation osseuse'] ?? null)) {
                    $localisation = $greffeData['Localisation'] ?? null;

                    $intervention->greffeOsseuse()->create([
                        'type' => $greffeData['Type'] ?? null,
                        'localisation' => is_array($localisation) ? json_encode($localisation) : $localisation,
                    ]);
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données de greffe osseuse : ' . $e->getMessage());
            }

            // == 13. SYSTÈME DE NAVIGATION ==
            try {
                $navData = $request->input('Système de navigation');

                if ($navData && $this->parseBoolean($navData['Utilisation'] ?? null)) {
                    $intervention->systemeNavigation()->create([
                        'systeme_utilise' => $navData['Système utilisé'] ?? null,
                        'numero_serie' => $navData['Numéro de série'] ?? null,
                        // Pour le genou uniquement
                        'chirurgie_mini_invasive' => $this->parseBoolean($navData['Chirurgie mini-invasive'] ?? null),
                    ]);
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données du système de navigation : ' . $e->getMessage());
            }

            // == 14. ANALGÉSIE ==
            try {
                $analgesieData = $request->input('Analgésie par infiltration locale ou loco-régionale');

                if ($analgesieData) {
                    $intervention->analgesie()->create([
                        'infiltration' => $this->parseBoolean($analgesieData['Infiltration'] ?? null),
                        'catheter_en_place' => $this->parseBoolean($analgesieData['Cathéter laissé en place'] ?? null),
                        'produit' => $analgesieData['Produit'] ?? null,
                    ]);
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données d\'analgésie : ' . $e->getMessage());
            }

            // == 15. PROPHYLAXIE ANTIBIOTIQUE ==
            try {
                $prophylaxieData = $request->input('Prophylaxie antibiotique');

                if ($prophylaxieData && $this->parseBoolean($prophylaxieData['Utilisée'] ?? null)) {
                    $intervention->prophylaxieAntibiotique()->create([
                        'antibiotique' => $prophylaxieData['Si oui préciser antibiotique'] ?? null,
                        'dose_par_jour' => $prophylaxieData['Dose /jour'] ?? null,
                        'debute_per_operatoire' => $this->parseBoolean($prophylaxieData['Débuté en per-opératoire'] ?? null),
                        'duree' => $prophylaxieData['Durée'] ?? null,
                    ]);
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données de prophylaxie antibiotique : ' . $e->getMessage());
            }

            // == 16. DIFFICULTÉS OPÉRATOIRES ==
            try {
                $difficulteData = $request->input('Difficulté opératoire');

                if ($difficulteData) {
                    $surComposante = $difficulteData['Sur'] ?? null;
                    if (is_array($surComposante)) {
                        $surComposante = json_encode($surComposante);
                    }

                    $nature = [];
                    $natureAutre = null;
                    $natureData = $difficulteData['Nature'] ?? null;

                    if (is_array($natureData)) {
                        if (isset($natureData['value'])) {
                            $nature = $natureData['value'];
                            $natureAutre = $natureData['autre'] ?? null;
                        } else {
                            $nature = $natureData;
                        }
                    }

                    $intervention->difficulteOperatoire()->create([
                        'sur_composante' => $surComposante,
                        'nature' => json_encode($nature),
                        'nature_autre' => $natureAutre,
                    ]);
                }
            } catch (\Exception $e) {
                throw new \Exception('Erreur lors de la création des données de difficultés opératoires : ' . $e->getMessage());
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Intervention enregistrée avec succès.',
                'data' => [
                    'intervention_id' => $intervention->id,
                    'patient_id' => $patient->id,
                    'surgeon_id' => $surgeon->id,
                ]
            ], 201);

        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();

            $errorCode = $e->errorInfo[1] ?? null;
            $errorMessage = 'Erreur de base de données. L\'enregistrement a échoué.';

            if ($errorCode === 1062) {
                $errorMessage = 'Cette intervention existe déjà (duplication détectée).';
            } elseif ($errorCode === 1452) {
                $errorMessage = 'Référence invalide : une clé étrangère n\'existe pas.';
            }

            Log::error('Erreur SQL store: ' . $e->getMessage(), [
                'code' => $errorCode,
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);

            return response()->json([
                'success' => false,
                'message' => $errorMessage,
                'error' => config('app.debug') ? $e->getMessage() : 'Erreur interne du serveur.',
                'error_code' => 'DB_ERROR'
            ], 500);

        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Erreur store formulaire: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => config('app.debug') ? $e->getTraceAsString() : 'Trace masquée'
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Une erreur est survenue lors de l\'enregistrement.',
                'error' => $e->getMessage(), // Le message est précis grâce aux exceptions lancées
                'error_location' => config('app.debug') ? [
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ] : null,
                'error_code' => 'GENERAL_ERROR'
            ], 500);
        }
    }
    //
    // =====================================================
    // MÉTHODES HELPERS À AJOUTER DANS VOTRE CONTROLLER
    // =====================================================

    /**
     * Parse une valeur booléenne depuis le formulaire
     */
    private function parseBoolean($value): bool
    {
        if (is_bool($value)) return $value;
        if (is_string($value)) {
            return in_array(strtolower($value), ['oui', 'yes', 'true', '1']);
        }
        return (bool) $value;
    }

    /**
     * Parse une valeur numérique
     */
    private function parseNumeric($value)
    {
        if (is_null($value) || $value === '') return null;
        if (is_numeric($value)) return floatval($value);
        // Remplacer la virgule par un point
        $cleaned = str_replace(',', '.', $value);
        return is_numeric($cleaned) ? floatval($cleaned) : null;
    }

    /**
     * Parse une date depuis différents formats
     */
    private function parseDate($value)
    {
        if (empty($value)) return null;

        // Si c'est déjà au format Y-m-d
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return $value;
        }

        // Format JJ-MM-AAAA ou JJ/MM/AAAA
        if (preg_match('/^(\d{2})[-\/](\d{2})[-\/](\d{4})$/', $value, $matches)) {
            return $matches[3] . '-' . $matches[2] . '-' . $matches[1];
        }

        // Tentative avec Carbon
        try {
            return \Carbon\Carbon::parse($value)->format('Y-m-d');
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Extrait les données restantes pour le champ JSON
     */
    private function extractAutresDonnees(Request $request, string $formType): array
    {
        $exclusions = [
            'surgeon_firebase_uid', 'surgeon_name', 'formType',
            'IDENTITE DU PATIENT', 'TERRAIN (S) - ANTECEDENTS', 'TERRAIN (S) - ANTECEDANTS',
            'DONNEES CLINIQUES', 'DONNEES PARACLINIQUES', 'Chirurgien', 'Anesthésie',
            'Suites opératoires et Résultats', 'En cas de révision', 'Greffe osseuse',
            'Système de navigation', 'Analgésie par infiltration locale ou loco-régionale',
            'Prophylaxie antibiotique', 'Difficulté opératoire', 'Implants utilisés',
            'DONNEES THERAPEUTIQUES', 'PTH (Arthroplastie totale) 1ère intention',
            'PTH (Arthroplastie totale) de reprise', 'PIH (Hémiarthroplastie)', 'Drainage'
        ];

        return \Illuminate\Support\Arr::except($request->all(), $exclusions);
    }

    /**
     * Crée les implants selon le type de formulaire
     */
    private function createImplants($intervention, Request $request, string $formType): void
    {
        if ($formType === 'hanche') {
            $this->createImplantsHanche($intervention, $request);
        } else {
            $this->createImplantsGenou($intervention, $request);
        }
    }

    /**
     * Crée les implants pour une prothèse de hanche
     */
    private function createImplantsHanche($intervention, Request $request): void
    {
        // PTH 1ère intention
        $pthData = $request->input('PTH (Arthroplastie totale) 1ère intention');

        if ($pthData) {
            // Traçabilité Cupule
            if ($cupuleData = $pthData['Traçabilité (Cupule)'] ?? null) {
                $this->createImplantFromTracabilite($intervention, 'Cupule', $cupuleData, [
                    'composant_acetabulaire' => $pthData['Composant acétabulaire'] ?? null,
                ]);
            }

            // Traçabilité Tige
            if ($tigeData = $pthData['Traçabilité (Tige)'] ?? null) {
                $this->createImplantFromTracabilite($intervention, 'Tige fémorale', $tigeData, [
                    'tige_femorale' => $pthData['Tige fémorale'] ?? null,
                ]);
            }
        }

        // PTH de reprise
        $pthRepriseData = $request->input('PTH (Arthroplastie totale) de reprise');

        if ($pthRepriseData) {
            if ($cupuleData = $pthRepriseData['Traçabilité (Cupule)'] ?? null) {
                $this->createImplantFromTracabilite($intervention, 'Cupule (reprise)', $cupuleData, [
                    'composant_acetabulaire' => $pthRepriseData['Composant acétabulaire'] ?? null,
                ]);
            }

            if ($tigeData = $pthRepriseData['Traçabilité (Tige)'] ?? null) {
                $this->createImplantFromTracabilite($intervention, 'Tige fémorale (reprise)', $tigeData, [
                    'tige_femorale' => $pthRepriseData['Tige fémorale'] ?? null,
                ]);
            }
        }

        // PIH
        $pihData = $request->input('PIH (Hémiarthroplastie)');

        if ($pihData) {
            if ($tracData = $pihData['Traçabilité (Cupule)'] ?? null) {
                // Extraire le type de PIH
                $typePih = null;
                $typePihAutre = null;
                $typeData = $pihData['Type'] ?? null;

                if (is_array($typeData)) {
                    $typePih = $typeData['value'] ?? null;
                    $typePihAutre = $typeData['autre'] ?? null;
                } else {
                    $typePih = $typeData;
                }

                $this->createImplantFromTracabilite($intervention, 'PIH', $tracData, [
                    'type_pih' => $typePih,
                    'type_pih_autre' => $typePihAutre,
                    'type_fixation' => $pihData['Type de fixation'] ?? null,
                    'diametre_tete_mm' => $pihData['Diamètre tête (mm)'] ?? null,
                    'taille_tige_mm' => $pihData['Taille de la tige (mm)'] ?? null,
                    'longueur_col_mm' => $pihData['Longueur du col (mm)'] ?? null,
                ]);
            }
        }
    }

    /**
     * Crée les implants pour une prothèse de genou
     */
    private function createImplantsGenou($intervention, Request $request): void
    {
        $implantsData = $request->input('Implants utilisés', []);

        // Composant fémoral
        if ($femoralData = $implantsData['Composant fémoral'] ?? null) {
            $tailleValue = null;
            $tailleAutre = null;
            $tailleData = $femoralData['Taille'] ?? null;

            if (is_array($tailleData)) {
                $tailleValue = $tailleData['value'] ?? null;
                $tailleAutre = $tailleData['autre'] ?? null;
            } else {
                $tailleValue = $tailleData;
            }

            $alliageValue = null;
            $alliageAutre = null;
            $alliageData = $femoralData['Alliage'] ?? null;

            if (is_array($alliageData)) {
                $alliageValue = $alliageData['value'] ?? null;
                $alliageAutre = $alliageData['autre'] ?? null;
            } else {
                $alliageValue = $alliageData;
            }

            $intervention->implants()->create([
                'type_composant' => 'Composant fémoral',
                'societe' => $femoralData['Société (Traçabilité)'] ?? null,
                'marque_modele' => $femoralData['Marque / Modèle'] ?? null,
                'reference' => $femoralData['Référence (Traçabilité)'] ?? null,
                'numero_lot' => $femoralData['Numéro de lot (Traçabilité)'] ?? null,
                'udi_di_pi' => $femoralData['UDI-DI / UDI-PI (Traçabilité)'] ?? null,
                'date_fabrication' => $this->parseDate($femoralData['Date de fabrication (Traçabilité)'] ?? null),
                'pays_fabrication' => $this->extractPays($femoralData['Pays de fabrication (Traçabilité)'] ?? null),
                'details_specifiques' => json_encode([
                    'taille' => $tailleValue,
                    'taille_autre' => $tailleAutre,
                    'fixation' => $femoralData['Fixation'] ?? null,
                    'alliage' => $alliageValue,
                    'alliage_autre' => $alliageAutre,
                ]),
            ]);
        }

        // Composant tibial
        if ($tibialData = $implantsData['Composant tibial (plateau)'] ?? null) {
            $tailleValue = null;
            $tailleAutre = null;
            $tailleData = $tibialData['Taille'] ?? null;

            if (is_array($tailleData)) {
                $tailleValue = $tailleData['value'] ?? null;
                $tailleAutre = $tailleData['autre'] ?? null;
            } else {
                $tailleValue = $tailleData;
            }

            $alliageValue = null;
            $alliageAutre = null;
            $alliageData = $tibialData['Alliage'] ?? null;

            if (is_array($alliageData)) {
                $alliageValue = $alliageData['value'] ?? null;
                $alliageAutre = $alliageData['autre'] ?? null;
            } else {
                $alliageValue = $alliageData;
            }

            $quilleValue = null;
            $quilleLongueur = null;
            $quilleData = $tibialData['Présence de quille'] ?? null;

            if (is_array($quilleData)) {
                $quilleValue = $this->parseBoolean($quilleData['value'] ?? null);
                $quilleLongueur = $quilleData['autre'] ?? null;
            } else {
                $quilleValue = $this->parseBoolean($quilleData);
            }

            $intervention->implants()->create([
                'type_composant' => 'Composant tibial',
                'societe' => null,
                'marque_modele' => $tibialData['Marque / Modèle'] ?? null,
                'reference' => null,
                'numero_lot' => null,
                'udi_di_pi' => null,
                'date_fabrication' => null,
                'pays_fabrication' => null,
                'details_specifiques' => json_encode([
                    'taille' => $tailleValue,
                    'taille_autre' => $tailleAutre,
                    'fixation' => $tibialData['Fixation'] ?? null,
                    'plateau' => $tibialData['Plateau'] ?? null,
                    'presence_quille' => $quilleValue,
                    'quille_longueur_mm' => $quilleLongueur,
                    'alliage' => $alliageValue,
                    'alliage_autre' => $alliageAutre,
                ]),
            ]);
        }

        // Insert (polyéthylène)
        if ($insertData = $implantsData['Insert (polyéthylène)'] ?? null) {
            $materiauValue = null;
            $materiauAutre = null;
            $materiauData = $insertData['Matériau'] ?? null;

            if (is_array($materiauData)) {
                $materiauValue = $materiauData['value'] ?? null;
                $materiauAutre = $materiauData['autre'] ?? null;
            } else {
                $materiauValue = $materiauData;
            }

            $intervention->implants()->create([
                'type_composant' => 'Insert',
                'societe' => $insertData['Société (Traçabilité)'] ?? null,
                'marque_modele' => null,
                'reference' => $insertData['Référence (Traçabilité)'] ?? null,
                'numero_lot' => $insertData['Numéro de lot (Traçabilité)'] ?? null,
                'udi_di_pi' => $insertData['UDI-DI / UDI-PI (Traçabilité)'] ?? null,
                'date_fabrication' => $this->parseDate($insertData['Date de fabrication (Traçabilité)'] ?? null),
                'pays_fabrication' => $this->extractPays($insertData['Pays de fabrication (Traçabilité)'] ?? null),
                'details_specifiques' => json_encode([
                    'materiau' => $materiauValue,
                    'materiau_autre' => $materiauAutre,
                    'epaisseur_mm' => $this->parseNumeric($insertData['Épaisseur (mm)'] ?? null),
                    'mobilite' => $insertData['Mobilité'] ?? null,
                    'design' => $insertData['Design'] ?? null,
                ]),
            ]);
        }

        // Composant patellaire
        if ($patellaData = $implantsData['Composant patellaire'] ?? null) {
            if ($this->parseBoolean($patellaData['Patella posée'] ?? null)) {
                $typeValue = null;
                $typeAutre = null;
                $typeData = $patellaData['Type'] ?? null;

                if (is_array($typeData)) {
                    $typeValue = $typeData['value'] ?? null;
                    $typeAutre = $typeData['autre'] ?? null;
                } else {
                    $typeValue = $typeData;
                }

                $intervention->implants()->create([
                    'type_composant' => 'Composant patellaire',
                    'societe' => null,
                    'marque_modele' => null,
                    'reference' => $patellaData['Réf. / Numéro de lot / UDI-PI'] ?? null,
                    'numero_lot' => null,
                    'udi_di_pi' => null,
                    'date_fabrication' => null,
                    'pays_fabrication' => null,
                    'details_specifiques' => json_encode([
                        'patella_posee' => true,
                        'type' => $typeValue,
                        'type_autre' => $typeAutre,
                        'fixation' => $patellaData['Fixation'] ?? null,
                        'diametre_mm' => $this->parseNumeric($patellaData['Diamètre (mm)'] ?? null),
                    ]),
                ]);
            }
        }
    }

    /**
     * Helper pour créer un implant depuis les données de traçabilité
     */
    private function createImplantFromTracabilite($intervention, string $typeComposant, array $tracData, array $detailsSupp = []): void
    {
        $paysFab = $this->extractPays($tracData['Pays de fabrication'] ?? null);

        $intervention->implants()->create([
            'type_composant' => $typeComposant,
            'societe' => $tracData['Société'] ?? null,
            'marque_modele' => $tracData['Marque de la prothèse'] ?? $tracData['Marque de la tige'] ?? null,
            'reference' => $tracData['Référence'] ?? null,
            'numero_lot' => $tracData['Numéro de lot'] ?? null,
            'udi_di_pi' => $tracData['UDI-DI / UDI-PI'] ?? null,
            'date_fabrication' => $this->parseDate($tracData['Date de fabrication'] ?? null),
            'pays_fabrication' => $paysFab,
            'details_specifiques' => !empty($detailsSupp) ? json_encode($detailsSupp) : null,
        ]);
    }

    /**
     * Extrait le pays de fabrication
     */
    private function extractPays($data): ?string
    {
        if (is_array($data)) {
            $value = $data['value'] ?? null;
            if ($value === 'autre' || $value === 'Autre (Préciser)') {
                return $data['autre'] ?? null;
            }
            return $value;
        }
        return $data;
    }

    /**
     * Crée les détails thérapeutiques selon le type de formulaire
     */
    private function createDetailsTherapeutiques($intervention, Request $request, string $formType): void
    {
        if ($formType === 'hanche') {
            $this->createDetailsTherapeutiquesHanche($intervention, $request);
        } else {
            $this->createDetailsTherapeutiquesGenou($intervention, $request);
        }
    }

    /**
     * Détails thérapeutiques pour la hanche
     */
    private function createDetailsTherapeutiquesHanche($intervention, Request $request): void
    {
        $pthData = $request->input('PTH (Arthroplastie totale) 1ère intention');
        $pthRepriseData = $request->input('PTH (Arthroplastie totale) de reprise');
        $pihData = $request->input('PIH (Hémiarthroplastie)');
        $donneesTherapeutiques = $request->input('DONNEES THERAPEUTIQUES.Implants', []);

        // Déterminer le type de prothèse (PTH ou PIH)
        $typeProthese = $donneesTherapeutiques['Type de prothèse'] ?? null;

        $detailsSupp = [];

        // Stocker le type de prothèse
        $detailsSupp['type_prothese_hanche'] = $typeProthese;

        // Si PIH
        if ($typeProthese === 'PIH' && $pihData) {
            $detailsSupp['pih'] = [
                'si_pih' => $donneesTherapeutiques['Si PIH'] ?? null,
            ];
        }

        // Si PTH
        if ($typeProthese === 'PTH' && $donneesTherapeutiques) {
            $detailsSupp['pth'] = [
                'si_pth' => $donneesTherapeutiques['Si PTH'] ?? null,
                'type_pth' => $donneesTherapeutiques['Type de PTH'] ?? null,
            ];
        }

        // Données PTH 1ère intention
        if ($pthData) {
            // Couple de frottement
            $coupleFrottement = null;
            $coupleData = $pthData['Couple de frottement'] ?? null;
            if (is_array($coupleData)) {
                $coupleFrottement = $coupleData[0] ?? null;
            } else {
                $coupleFrottement = $coupleData;
            }

            // Type cupule
            $typeCupule = null;
            $composantAcetab = $pthData['Composant acétabulaire'] ?? [];
            if ($composantAcetab) {
                $typeCupule = $composantAcetab['Type de fixation'] ?? null;

                // Stocker tous les détails du composant acétabulaire
                $detailsSupp['composant_acetabulaire'] = [
                    'type_fixation' => $composantAcetab['Type de fixation'] ?? null,
                    'alliage' => $this->extractValueAutre($composantAcetab['Alliage'] ?? null),
                    'revetement_surface' => $this->extractValueAutre($composantAcetab['Revêtement de surface'] ?? null),
                    'reconstruction' => $this->extractValueAutre($composantAcetab['Reconstruction acétabulaire'] ?? null),
                    'taille_cupule' => $composantAcetab['Taille : cupule'] ?? null,
                    'materiaux_insert' => $this->extractValueAutre($composantAcetab['Materiaux Insert'] ?? null),
                ];
            }

            // Tige fémorale
            $tigeCollerette = null;
            $tigeFixation = null;
            $tigeFemorale = $pthData['Tige fémorale'] ?? [];
            if ($tigeFemorale) {
                $tigeCollerette = $tigeFemorale['Collerette'] ?? null;
                $tigeFixation = $tigeFemorale['Type de fixation'] ?? null;

                $detailsSupp['tige_femorale'] = [
                    'collerette' => $tigeCollerette,
                    'taille' => $tigeFemorale['Taille'] ?? null,
                    'longueur_col' => $tigeFemorale['Longueur du col'] ?? null,
                    'tete' => $this->extractValueAutre($tigeFemorale['Tête'] ?? null),
                    'type_fixation' => $tigeFixation,
                    'alliage' => $this->extractValueAutre($tigeFemorale['Alliage/Tige femorale'] ?? null),
                    'revetement_surface' => $this->extractValueAutre($tigeFemorale['Revêtement de surface'] ?? null),
                    'position' => $this->extractValueAutre($tigeFemorale['Position de la tige'] ?? null),
                ];
            }

            // Synthèse fémorale complémentaire
            $syntheseFemorale = $pthData['Synthèse fémorale complémentaire'] ?? [];
            if ($syntheseFemorale) {
                $detailsSupp['synthese_femorale'] = [
                    'type' => $this->extractValueAutre($syntheseFemorale['Type'] ?? null),
                    'drainage' => $this->parseBoolean($syntheseFemorale['Drainage'] ?? null),
                ];
            }

            // Resurfaçage
            $resurfacage = false;
            $resurfData = $pthData['Resurfaçage'] ?? null;
            if (is_array($resurfData)) {
                $resurfacage = $this->parseBoolean($resurfData[0] ?? null);
            } else {
                $resurfacage = $this->parseBoolean($resurfData);
            }

            $intervention->detailsTherapeutiquesImplants()->updateOrCreate(
                ['intervention_id' => $intervention->id],
                [
                    'couple_frottement' => $coupleFrottement,
                    'type_cupule' => $typeCupule,
                    'tige_femoral_collerette' => $tigeCollerette,
                    'tige_femoral_fixation' => $tigeFixation,
                    'resurfacage' => $resurfacage,
                    'details_supplementaires' => json_encode($detailsSupp),
                ]
            );

            return;
        }

        // PTH de reprise
        if ($pthRepriseData) {
            $detailsSupp['type_tige'] = $pthRepriseData['Type de tige'] ?? null;
            $detailsSupp['type_armature'] = $this->extractValueAutre($pthRepriseData['Type d\'armature cotyloïdienne'] ?? null);

            // Composant acétabulaire reprise
            $composantAcetab = $pthRepriseData['Composant acétabulaire'] ?? [];
            if ($composantAcetab) {
                $detailsSupp['composant_acetabulaire_reprise'] = [
                    'type_fixation' => $composantAcetab['Type de fixation'] ?? null,
                    'alliage' => $this->extractValueAutre($composantAcetab['Alliage'] ?? null),
                    'revetement_surface' => $this->extractValueAutre($composantAcetab['Revêtement de surface'] ?? null),
                    'reconstruction' => $this->extractValueAutre($composantAcetab['Reconstruction acétabulaire'] ?? null),
                    'taille_cupule' => $composantAcetab['Taille : cupule'] ?? null,
                    'materiaux_insert' => $this->extractValueAutre($composantAcetab['Materiaux Insert'] ?? null),
                ];
            }

            // Tige fémorale reprise
            $tigeFemorale = $pthRepriseData['Tige fémorale'] ?? [];
            $tigeCollerette = null;
            $tigeFixation = null;
            if ($tigeFemorale) {
                $tigeCollerette = $tigeFemorale['Collerette'] ?? null;
                $tigeFixation = $tigeFemorale['Type de fixation'] ?? null;

                $detailsSupp['tige_femorale_reprise'] = [
                    'collerette' => $tigeCollerette,
                    'taille' => $tigeFemorale['Taille'] ?? null,
                    'longueur_col' => $tigeFemorale['Longueur du col'] ?? null,
                    'tete' => $this->extractValueAutre($tigeFemorale['Tête'] ?? null),
                    'type_fixation' => $tigeFixation,
                    'alliage' => $this->extractValueAutre($tigeFemorale['Alliage/Tige femorale'] ?? null),
                    'revetement_surface' => $this->extractValueAutre($tigeFemorale['Revêtement de surface'] ?? null),
                    'position' => $this->extractValueAutre($tigeFemorale['Position de la tige'] ?? null),
                ];
            }

            // Synthèse fémorale complémentaire reprise
            $syntheseFemorale = $pthRepriseData['Synthèse fémorale complémentaire'] ?? [];
            if ($syntheseFemorale) {
                $detailsSupp['synthese_femorale_reprise'] = [
                    'type' => $this->extractValueAutre($syntheseFemorale['Type'] ?? null),
                    'drainage' => $this->parseBoolean($syntheseFemorale['Drainage'] ?? null),
                ];
            }

            // Resurfaçage reprise
            $resurfacage = false;
            $resurfData = $pthRepriseData['Resurfaçage'] ?? null;
            if (is_array($resurfData)) {
                $resurfacage = $this->parseBoolean($resurfData[0] ?? null);
            } else {
                $resurfacage = $this->parseBoolean($resurfData);
            }

            $intervention->detailsTherapeutiquesImplants()->updateOrCreate(
                ['intervention_id' => $intervention->id],
                [
                    'couple_frottement' => null,
                    'type_cupule' => $composantAcetab['Type de fixation'] ?? null,
                    'tige_femoral_collerette' => $tigeCollerette,
                    'tige_femoral_fixation' => $tigeFixation,
                    'resurfacage' => $resurfacage,
                    'details_supplementaires' => json_encode($detailsSupp),
                ]
            );

            return;
        }

        // PIH seulement
        if ($pihData) {
            $detailsSupp['pih_details'] = [
                'type' => $this->extractValueAutre($pihData['Type'] ?? null),
                'type_fixation' => $pihData['Type de fixation'] ?? null,
                'diametre_tete_mm' => $this->parseNumeric($pihData['Diamètre tête (mm)'] ?? null),
                'taille_tige_mm' => $this->parseNumeric($pihData['Taille de la tige (mm)'] ?? null),
                'longueur_col_mm' => $this->parseNumeric($pihData['Longueur du col (mm)'] ?? null),
            ];

            $intervention->detailsTherapeutiquesImplants()->updateOrCreate(
                ['intervention_id' => $intervention->id],
                [
                    'couple_frottement' => null,
                    'type_cupule' => null,
                    'tige_femoral_collerette' => null,
                    'tige_femoral_fixation' => $pihData['Type de fixation'] ?? null,
                    'resurfacage' => false,
                    'details_supplementaires' => json_encode($detailsSupp),
                ]
            );
        }
    }

    /**
     * Détails thérapeutiques pour le genou
     */
    private function createDetailsTherapeutiquesGenou($intervention, Request $request): void
    {
        $typeArthroplastieData = $request->input('DONNEES THERAPEUTIQUES.Type d\'arthroplastie', []);
        $implantsData = $request->input('Implants utilisés', []);
        $drainageData = $request->input('Drainage');

        // Type d'arthroplastie
        $typeArthroplastie = $typeArthroplastieData['Type'] ?? null;

        // Si prothèse primaire
        $typeProthesePrimaire = null;
        $typeProthesePrimaireAutre = null;
        $siPrimaireData = $typeArthroplastieData['Si prothèse primaire'] ?? null;

        if (is_array($siPrimaireData)) {
            $typeProthesePrimaire = $siPrimaireData['value'] ?? null;
            $typeProthesePrimaireAutre = $siPrimaireData['autre'] ?? null;
        } else {
            $typeProthesePrimaire = $siPrimaireData;
        }

        // Composant fémoral
        $femoralData = $implantsData['Composant fémoral'] ?? [];
        $fixationFemoral = $femoralData['Fixation'] ?? null;

        // Composant tibial
        $tibialData = $implantsData['Composant tibial (plateau)'] ?? [];
        $fixationTibial = $tibialData['Fixation'] ?? null;

        // Insert
        $insertData = $implantsData['Insert (polyéthylène)'] ?? [];
        $mobiliteInsert = $insertData['Mobilité'] ?? null;
        $designInsert = $insertData['Design'] ?? null;

        // Composant patellaire
        $patellaData = $implantsData['Composant patellaire'] ?? [];
        $patellaPosee = $this->parseBoolean($patellaData['Patella posée'] ?? null);

        // Détails supplémentaires
        $detailsSupp = [
            'type_prothese_primaire_autre' => $typeProthesePrimaireAutre,
            'composant_femoral' => [
                'marque_modele' => $femoralData['Marque / Modèle'] ?? null,
                'taille' => $this->extractValueAutre($femoralData['Taille'] ?? null),
                'alliage' => $this->extractValueAutre($femoralData['Alliage'] ?? null),
            ],
            'composant_tibial' => [
                'marque_modele' => $tibialData['Marque / Modèle'] ?? null,
                'taille' => $this->extractValueAutre($tibialData['Taille'] ?? null),
                'plateau' => $tibialData['Plateau'] ?? null,
                'presence_quille' => $this->extractValueAutre($tibialData['Présence de quille'] ?? null),
                'alliage' => $this->extractValueAutre($tibialData['Alliage'] ?? null),
            ],
            'insert' => [
                'materiau' => $this->extractValueAutre($insertData['Matériau'] ?? null),
                'epaisseur_mm' => $this->parseNumeric($insertData['Épaisseur (mm)'] ?? null),
            ],
            'composant_patellaire' => [
                'type' => $this->extractValueAutre($patellaData['Type'] ?? null),
                'fixation' => $patellaData['Fixation'] ?? null,
                'diametre_mm' => $this->parseNumeric($patellaData['Diamètre (mm)'] ?? null),
            ],
            'drainage' => is_array($drainageData)
                ? $this->parseBoolean($drainageData[0] ?? null)
                : $this->parseBoolean($drainageData),
        ];

        $intervention->detailsTherapeutiquesImplants()->updateOrCreate(
            ['intervention_id' => $intervention->id],
            [
                'type_arthroplastie' => $typeArthroplastie,
                'type_prothese_primaire' => $typeProthesePrimaire,
                'fixation_femoral' => $fixationFemoral,
                'fixation_tibial' => $fixationTibial,
                'mobilite_insert' => $mobiliteInsert,
                'design_insert' => $designInsert,
                'patella_posee' => $patellaPosee,
                'details_supplementaires' => json_encode($detailsSupp),
            ]
        );
    }

    /**
     * Extrait value et autre d'un champ radio_with_text
     */
    private function extractValueAutre($data): ?array
    {
        if (is_null($data)) return null;

        if (is_array($data)) {
            return [
                'value' => $data['value'] ?? null,
                'autre' => $data['autre'] ?? null,
            ];
        }

        return ['value' => $data, 'autre' => null];
    }
    //

    /**
     * Affiche une intervention spécifique avec toutes ses données formatées.
     * VERSION COMPLÈTE ET CORRIGÉE
     */
    public function show(Request $request, $id)
    {
        $surgeon = $request->attributes->get('surgeon');

        if (!$surgeon) {
            return response()->json([
                'success' => false,
                'message' => 'Chirurgien non authentifié'
            ], 401);
        }

        $intervention = Intervention::where('surgeon_id', $surgeon->id)
            ->with([
                'patient', 'antecedentsTerrains', 'donneesCliniques', 
                'donneesBiologiques', 'imageriesDiagnostics', 'implants', 
                'donneesAnesthesiques', 'suitesOperatoires', 'revision', 
                'detailsTherapeutiquesImplants', 'greffeOsseuse', 
                'systemeNavigation', 'analgesie', 'prophylaxieAntibiotique', 
                'difficulteOperatoire'
            ])
            ->find($id);

        if (!$intervention) {
            return response()->json([
                'success' => false,
                'message' => 'Intervention non trouvée'
            ], 404);
        }

        $response = [
            'id' => $intervention->id,
            'formType' => $intervention->type_prothese,
            'statut' => $this->determineStatus($intervention),
            'IDENTITE DU PATIENT' => $this->buildPatientIdentitySection($intervention),
            'TERRAIN (S) - ANTECEDENTS' => $this->buildAntecedentsSection($intervention),
            'TERRAIN (S) - ANTECEDANTS' => $this->buildAntecedentsSection($intervention),
            'DONNEES CLINIQUES' => $this->buildClinicalDataSection($intervention),
            'DONNEES PARACLINIQUES' => $this->buildParaclinicalDataSection($intervention),
            'DONNEES THERAPEUTIQUES' => $this->buildTherapeuticDataSection($intervention),
            'Chirurgien' => $this->buildSurgeonSection($intervention),
            'Anesthésie' => $this->buildAnesthesiaSection($intervention),
            'Analgésie par infiltration locale ou loco-régionale' => $this->buildAnalgesiaSection($intervention),
            'Prophylaxie antibiotique' => $this->buildAntibioticProphylaxisSection($intervention),
            'Greffe osseuse' => $this->buildBoneGraftSection($intervention),
            'Système de navigation' => $this->buildNavigationSystemSection($intervention),
            'Difficulté opératoire' => $this->buildOperativeDifficultySection($intervention),
            'implants_traçabilité' => $this->buildImplantsTraceabilitySection($intervention),
            'Suites opératoires et Résultats' => $this->buildPostOperativeSection($intervention),
            'En cas de révision' => $this->buildRevisionSection($intervention),
        ];

        // Sections spécifiques selon le type
        if ($intervention->type_prothese === 'hanche') {
            $response['PIH (Hémiarthroplastie)'] = $this->buildPIHSection($intervention);
            $response['PTH (Arthroplastie totale) 1ère intention'] = $this->buildPTHPremierSection($intervention);
            $response['PTH (Arthroplastie totale) de reprise'] = $this->buildPTHRepriseSection($intervention);
        } else {
            $response['DONNEES THERAPEUTIQUES']['Type d\'arthroplastie'] = $this->buildKneeArthroplastySection($intervention);
            $response['Implants utilisés'] = $this->buildKneeImplantsSection($intervention);
            $response['Drainage'] = $this->buildDrainageSection($intervention);
        }

        // Fusion avec données additionnelles
        $additionalData = $this->decodeJsonField($intervention->donnees_formulaire_autres);

        return response()->json(array_merge($response, $additionalData));
    }

    /**
     * Détermine le statut de l'intervention
     */
    private function determineStatus($intervention): string
    {
        $required = ['patient', 'donneesCliniques', 'donneesAnesthesiques'];
        foreach ($required as $relation) {
            if (!$intervention->$relation) return 'incomplete';
        }
        return 'complete';
    }

    /**
     * Helper pour décoder un champ JSON
     */
    private function decodeJsonField($value, $default = [])
    {
        if (is_string($value)) {
            $decoded = json_decode($value, true);
            return $decoded !== null ? $decoded : $default;
        }
        return is_array($value) ? $value : $default;
    }

    /**
     * Helper pour accéder aux propriétés d'une relation nullable
     */
    private function safeGet($relation, $property, $default = null)
    {
        return $relation?->{$property} ?? $default;
    }

    /**
     * Formate une date pour l'affichage
     */
    private function formatDate($date): ?string
    {
        if (!$date) return null;
        if (is_string($date)) return $date;
        return $date->format('Y-m-d');
    }

    /**
     * Formate un booléen en Oui/Non
     */
    private function formatBool($value): string
    {
        return $value ? 'Oui' : 'Non';
    }

    /**
     * Construction de la section identité du patient
     */
    private function buildPatientIdentitySection($intervention): array
    {
        $patient = $intervention->patient;

        // Récupérer l'hôpital depuis l'intervention
        $hopital = $intervention->hopital_etablissement;
        $hopitalValue = $hopital;
        $hopitalAutre = null;

        // Vérifier si c'est une valeur prédéfinie ou "autre"
        $hopitauxConnus = ['CHUB', 'CHUP', 'CHUSS', 'CHUT', 'CHUYO', 'CM la Grâce'];
        if ($hopital && !in_array($hopital, $hopitauxConnus)) {
            $hopitalValue = 'autre';
            $hopitalAutre = $hopital;
        }

        return [
            'Nom' => $patient->nom,
            'Prénom(s)' => $patient->prenom,
            'Date de naissance (JJ-MM-AAAA)' => $this->formatDate($patient->date_naissance),
            'Sexe' => $patient->sexe,
            'Profession' => [
                'value' => $patient->profession,
                'autre' => $patient->profession_autre,
            ],
            'Adresse du patient' => $patient->adresse,
            'Téléphone' => $patient->telephone,
            'Hôpital / numéro d\'établissement' => [
                'value' => $hopitalValue,
                'autre' => $hopitalAutre,
            ],
            'Numéro Identifiant' => $patient->numero_identifiant,
        ];
    }

    /**
     * Construction de la section antécédents
     */
    private function buildAntecedentsSection($intervention): ?array
    {
        $ant = $intervention->antecedentsTerrains;
        if (!$ant) return null;

        return [
            'Antécédents médicaux' => $this->decodeJsonField($ant->antecedents_medicaux, []),
            'Sérologie rétroviral' => $ant->serologie_retroviral,
            'Traumatisme' => $this->formatBool($ant->traumatisme),
            'Corticothérapie' => $this->formatBool($ant->corticotherapie),
            'Intervention chirurgicale sur cette hanche' => [
                'value' => $ant->intervention_anterieure,
                'autre' => $ant->intervention_anterieure_autre,
            ],
            'Intervention chirurgicale sur ce genou' => [
                'value' => $ant->intervention_anterieure,
                'autre' => $ant->intervention_anterieure_autre,
            ],
        ];
    }

    /**
     * Construction de la section données cliniques
     */
    private function buildClinicalDataSection($intervention): ?array
    {
        $clin = $intervention->donneesCliniques;
        if (!$clin) return null;

        return [
            'Poids (kg)' => $clin->poids_kg,
            'Taille (cm)' => $clin->taille_cm,
            'IMC (kg/m²)' => $clin->imc,
            'Côté' => $clin->cote_opere,
            'Boiterie' => $this->formatBool($clin->boiterie),
            'Impotence fonctionnelle' => [
                'value' => $this->formatBool($clin->impotence_fonctionnelle),
                'conditional_value' => $clin->impotence_fonctionnelle_precision,
            ],
            'PMA' => $clin->pma_pre_operatoire,
            'Score de Parker' => $clin->score_parker,
            'ILMI' => [
                'value' => $this->formatBool($clin->ilmi),
                'autre' => $clin->ilmi_cm,
            ],
        ];
    }

    /**
     * Construction de la section données paracliniques
     */
    private function buildParaclinicalDataSection($intervention): array
    {
        $bio = $intervention->donneesBiologiques;
        $img = $intervention->imageriesDiagnostics;

        return [
            'Biologie' => $bio ? [
                'Globules Blancs (10³/mm³)' => $bio->globules_blancs,
                'Neutrophiles (10³/mm³)' => $bio->neutrophiles,
                'Hémoglobine (g/dl)' => $bio->hemoglobine,
                'Plaquettes (10³/mm³)' => $bio->plaquettes,
                'Groupe Sanguin' => $bio->groupe_sanguin,
                'Rhésus' => $bio->rhesus,
                'Créatininémie (µmol/L)' => $bio->creatinine,
                'Glycémie (mmol/L)' => $bio->glycemie,
                'CRP (mg/L)' => $bio->crp,
                'Electrophorèse de l\'hémoglobine' => $bio->electrophorese_hemoglobine,
                'Procalcitonine (ng/mL)' => $bio->procalcitonine,
                'TP/TCK' => $bio->tp_tck,
                'INR' => $bio->inr,
                'Autre (préciser)' => $bio->autre_biologie,
            ] : null,
            'Imagerie (Diagnostic)' => $img ? [
                'Techniques' => $this->decodeJsonField($img->techniques, []),
                'Résultats (cocher le principal)' => [
                    'value' => $img->resultat_principal,
                    'autre' => $img->resultat_principal_autre,
                ],
                'Diagnostic (cocher le principal)' => [
                    'value' => $img->resultat_principal,
                    'autre' => $img->resultat_principal_autre,
                ],
            ] : null,
        ];
    }

    /**
     * Construction de la section données thérapeutiques
     */
    private function buildTherapeuticDataSection($intervention): array
    {
        $voieAbord = $intervention->voie_dabord;
        $voieAbordArray = null;

        // Pour le genou, la voie d'abord peut être un JSON
        if ($intervention->type_prothese === 'genou') {
            $voieAbordArray = $this->decodeJsonField($voieAbord, []);
            if (empty($voieAbordArray) && $voieAbord) {
                $voieAbordArray = [$voieAbord];
            }
        }

        $details = $intervention->detailsTherapeutiquesImplants;
        $detailsSupp = $details ? $this->decodeJsonField($details->details_supplementaires, []) : [];

        return [
            'Chirurgicales' => [
                'Date intervention' => $this->formatDate($intervention->date_intervention),
                'Heure début' => $intervention->heure_debut,
                'Fin' => $intervention->heure_fin,
                'Voie d\'abord de la hanche' => [
                    'value' => $intervention->type_prothese === 'hanche' ? $voieAbord : null,
                    'autre' => $intervention->voie_dabord_autre,
                ],
                'Voie d\'abord' => $voieAbordArray,
            ],
            'Implants' => [
                'Type de prothèse' => $detailsSupp['type_prothese_hanche'] ?? null,
                'Si PIH' => $detailsSupp['pih']['si_pih'] ?? null,
                'Si PTH' => $detailsSupp['pth']['si_pth'] ?? null,
                'Type de PTH' => $detailsSupp['pth']['type_pth'] ?? null,
            ],
        ];
    }

    /**
     * Construction de la section chirurgien
     */
    private function buildSurgeonSection($intervention): array
    {
        return [
            'Identifiant/code' => $intervention->chirurgien_identifiant,
        ];
    }

    /**
     * Construction de la section anesthésie
     */
    private function buildAnesthesiaSection($intervention): ?array
    {
        $anesth = $intervention->donneesAnesthesiques;
        if (!$anesth) return null;

        return [
            'Type' => [
                'value' => $anesth->type_anesthesie,
                'autre' => $anesth->type_anesthesie_autre,
            ],
            'Classification ASA' => $anesth->classification_asa,
            'Complications per-anesthésiques' => [
                'value' => $this->formatBool($anesth->complications_per_anesthesiques),
                'autre' => $anesth->complications_per_anesthesiques_details,
            ],
        ];
    }

    /**
     * Construction de la section analgésie
     */
    private function buildAnalgesiaSection($intervention): ?array
    {
        $analg = $intervention->analgesie;
        if (!$analg) return null;

        return [
            'Infiltration' => $this->formatBool($analg->infiltration),
            'Cathéter laissé en place' => $this->formatBool($analg->catheter_en_place),
            'Produit' => $analg->produit,
        ];
    }

    /**
     * Construction de la section prophylaxie antibiotique
     */
    private function buildAntibioticProphylaxisSection($intervention): array
    {
        $proph = $intervention->prophylaxieAntibiotique;

        if (!$proph) {
            return ['Utilisée' => 'Non'];
        }

        return [
            'Utilisée' => 'Oui',
            'Si oui préciser antibiotique' => $proph->antibiotique,
            'Dose /jour' => $proph->dose_par_jour,
            'Débuté en per-opératoire' => $this->formatBool($proph->debute_per_operatoire),
            'Durée' => $proph->duree,
        ];
    }

    /**
     * Construction de la section greffe osseuse
     */
    private function buildBoneGraftSection($intervention): array
    {
        $greffe = $intervention->greffeOsseuse;

        if (!$greffe) {
            return ['Transplantation osseuse' => 'Non'];
        }

        $localisation = $greffe->localisation;
        if (is_string($localisation)) {
            $localisation = $this->decodeJsonField($localisation, $localisation);
        }

        return [
            'Transplantation osseuse' => 'Oui',
            'Type' => $greffe->type,
            'Localisation' => $localisation,
        ];
    }

    /**
     * Construction de la section système de navigation
     */
    private function buildNavigationSystemSection($intervention): array
    {
        $nav = $intervention->systemeNavigation;

        if (!$nav) {
            return ['Utilisation' => 'Non'];
        }

        $result = [
            'Utilisation' => 'Oui',
            'Système utilisé' => $nav->systeme_utilise,
            'Numéro de série' => $nav->numero_serie,
        ];

        // Champ spécifique au genou
        if ($intervention->type_prothese === 'genou' && isset($nav->chirurgie_mini_invasive)) {
            $result['Chirurgie mini-invasive'] = $this->formatBool($nav->chirurgie_mini_invasive);
        }

        return $result;
    }

    /**
     * Construction de la section difficulté opératoire
     */
    private function buildOperativeDifficultySection($intervention): ?array
    {
        $diff = $intervention->difficulteOperatoire;
        if (!$diff) return null;

        $surComposante = $diff->sur_composante;
        if (is_string($surComposante)) {
            $decoded = $this->decodeJsonField($surComposante, null);
            $surComposante = $decoded ?: $surComposante;
        }

        return [
            'Sur' => $surComposante,
            'Nature' => [
                'value' => $this->decodeJsonField($diff->nature, []),
                'autre' => $diff->nature_autre,
            ],
        ];
    }

    /**
     * Construction de la section traçabilité des implants
     */
    private function buildImplantsTraceabilitySection($intervention): array
    {
        return $intervention->implants->map(function ($implant) {
            $detailsSpec = $this->decodeJsonField($implant->details_specifiques, []);

            return [
                'type_composant' => $implant->type_composant,
                'societe' => $implant->societe,
                'marque_modele' => $implant->marque_modele,
                'reference' => $implant->reference,
                'numero_lot' => $implant->numero_lot,
                'udi_di_pi' => $implant->udi_di_pi,
                'date_fabrication' => $this->formatDate($implant->date_fabrication),
                'pays_fabrication' => $implant->pays_fabrication,
                'details_specifiques' => $detailsSpec,
            ];
        })->toArray();
    }

    /**
     * Construction de la section suites opératoires
     */
    private function buildPostOperativeSection($intervention): ?array
    {
        $suites = $intervention->suitesOperatoires;
        if (!$suites) return null;

        return [
            'Délai de mise en charge (jours)' => $suites->delai_mise_en_charge_jours,
            'Délai d\'appui total (mois)' => $suites->delai_appui_total_mois,
            'Durée de séjour hospitalier' => $suites->duree_sejour_hospitalier,
            'Rééducation fonctionnelle' => [
                'value' => $this->formatBool($suites->reeducation_fonctionnelle ?? false),
                'autre' => $suites->reeducation_nb_seances ?? null,
            ],
            'Complications per opératoire' => $this->decodeJsonField($suites->complications_per_operatoire, []),
            'Suites opératoires immédiates' => $suites->suites_immediates,
            'Complications immédiates' => $this->decodeJsonField($suites->complications_immediates, []),
            'Complications secondaires' => $this->decodeJsonField($suites->complications_secondaires, []),
            'Complications tardives' => $this->decodeJsonField($suites->complications_tardives, []),
            'Si descellement' => [
                'Type' => $suites->descellement_type ?? null,
                'Siège' => $suites->descellement_siege ?? null,
            ],
            'Prise en charge des frais de soins' => [
                'value' => $suites->prise_en_charge_frais,
                'autre' => $suites->prise_en_charge_frais_autre,
            ],
            'Dernier recul (mois)' => $suites->dernier_recul_mois,
            'Appréciation du patient lui-même : EVA' => $suites->eva_patient,
            'Correction d\'un raccourcissement' => [
                'value' => $this->formatBool($suites->correction_raccourcissement ?? false),
                'autre' => $suites->correction_raccourcissement_cm ?? null,
            ],
            'Correction d\'une boiterie' => $this->formatBool($suites->correction_boiterie ?? false),
            'Evaluation fonctionnelle : PMA' => $suites->pma_post_operatoire,
        ];
    }

    /**
     * Construction de la section révision
     */
    private function buildRevisionSection($intervention): ?array
    {
        $rev = $intervention->revision;
        if (!$rev) return null;

        return [
            'Type de révision' => [
                'Type' => $this->decodeJsonField($rev->types_revision, []),
                'Lequel(s) / Autre (préciser)' => $rev->types_revision_autre,
            ],
            'Motif de la révision' => [
                'Motif' => $this->decodeJsonField($rev->motifs_revision, []),
                'Fracture Spécifiée / Autre' => $rev->motifs_revision_autre,
            ],
        ];
    }

    // =====================================================
    // MÉTHODES SPÉCIFIQUES POUR LA HANCHE
    // À ajouter dans le Controller
    // =====================================================

    /**
     * Construction de la section PIH (Hémiarthroplastie)
     */
    private function buildPIHSection($intervention): ?array
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        if (!$details) return null;

        $detailsSupp = $this->decodeJsonField($details->details_supplementaires, []);
        $pihDetails = $detailsSupp['pih_details'] ?? [];

        if (empty($pihDetails)) return null;

        // Récupérer l'implant PIH
        $pihImplant = $intervention->implants->firstWhere('type_composant', 'PIH');
        $implantDetails = $pihImplant ? $this->decodeJsonField($pihImplant->details_specifiques, []) : [];

        return [
            'Type' => [
                'value' => $pihDetails['type']['value'] ?? null,
                'autre' => $pihDetails['type']['autre'] ?? null,
            ],
            'Type de fixation' => $pihDetails['type_fixation'] ?? $implantDetails['type_fixation'] ?? null,
            'Diamètre tête (mm)' => $pihDetails['diametre_tete_mm'] ?? $implantDetails['diametre_tete_mm'] ?? null,
            'Taille de la tige (mm)' => $pihDetails['taille_tige_mm'] ?? $implantDetails['taille_tige_mm'] ?? null,
            'Longueur du col (mm)' => $pihDetails['longueur_col_mm'] ?? $implantDetails['longueur_col_mm'] ?? null,
            'Traçabilité (Cupule)' => $pihImplant ? [
                'Société' => $pihImplant->societe,
                'Marque de la prothèse' => $pihImplant->marque_modele,
                'Référence' => $pihImplant->reference,
                'Numéro de lot' => $pihImplant->numero_lot,
                'UDI-DI / UDI-PI' => $pihImplant->udi_di_pi,
                'Date de fabrication' => $this->formatDate($pihImplant->date_fabrication),
                'Pays de fabrication' => [
                    'value' => $pihImplant->pays_fabrication,
                    'autre' => null,
                ],
            ] : null,
        ];
    }

    /**
     * Construction de la section PTH 1ère intention
     */
    private function buildPTHPremierSection($intervention): ?array
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        if (!$details) return null;

        $detailsSupp = $this->decodeJsonField($details->details_supplementaires, []);

        // Vérifier si c'est bien une PTH 1ère intention
        $typePth = $detailsSupp['pth']['si_pth'] ?? null;
        if ($typePth !== '1ère intention' && empty($detailsSupp['composant_acetabulaire'])) {
            return null;
        }

        // Récupérer les implants
        $cupuleImplant = $intervention->implants->firstWhere('type_composant', 'Cupule');
        $tigeImplant = $intervention->implants->firstWhere('type_composant', 'Tige fémorale');

        $compAcetab = $detailsSupp['composant_acetabulaire'] ?? [];
        $tigeFem = $detailsSupp['tige_femorale'] ?? [];
        $synthFem = $detailsSupp['synthese_femorale'] ?? [];

        return [
            'Couple de frottement' => $details->couple_frottement,
            'Composant acétabulaire' => [
                'Type de fixation' => $compAcetab['type_fixation'] ?? $details->type_cupule,
                'Alliage' => $compAcetab['alliage'] ?? null,
                'Revêtement de surface' => $compAcetab['revetement_surface'] ?? null,
                'Reconstruction acétabulaire' => $compAcetab['reconstruction'] ?? null,
                'Taille : cupule' => $compAcetab['taille_cupule'] ?? null,
                'Materiaux Insert' => $compAcetab['materiaux_insert'] ?? null,
            ],
            'Traçabilité (Cupule)' => $cupuleImplant ? [
                'Société' => $cupuleImplant->societe,
                'Marque de la prothèse' => $cupuleImplant->marque_modele,
                'Référence' => $cupuleImplant->reference,
                'Numéro de lot' => $cupuleImplant->numero_lot,
                'UDI-DI / UDI-PI' => $cupuleImplant->udi_di_pi,
                'Date de fabrication' => $this->formatDate($cupuleImplant->date_fabrication),
                'Pays de fabrication' => [
                    'value' => $cupuleImplant->pays_fabrication,
                    'autre' => null,
                ],
            ] : null,
            'Tige fémorale' => [
                'Collerette' => $tigeFem['collerette'] ?? $details->tige_femoral_collerette,
                'Taille' => $tigeFem['taille'] ?? null,
                'Longueur du col' => $tigeFem['longueur_col'] ?? null,
                'Tête' => $tigeFem['tete'] ?? null,
                'Type de fixation' => $tigeFem['type_fixation'] ?? $details->tige_femoral_fixation,
                'Alliage/Tige femorale' => $tigeFem['alliage'] ?? null,
                'Revêtement de surface' => $tigeFem['revetement_surface'] ?? null,
                'Position de la tige' => $tigeFem['position'] ?? null,
            ],
            'Synthèse fémorale complémentaire' => [
                'Type' => $synthFem['type'] ?? null,
                'Drainage' => isset($synthFem['drainage']) ? $this->formatBool($synthFem['drainage']) : null,
            ],
            'Traçabilité (Tige)' => $tigeImplant ? [
                'Société' => $tigeImplant->societe,
                'Marque de la tige' => $tigeImplant->marque_modele,
                'Référence' => $tigeImplant->reference,
                'Numéro de lot' => $tigeImplant->numero_lot,
                'UDI-DI / UDI-PI' => $tigeImplant->udi_di_pi,
                'Date de fabrication' => $this->formatDate($tigeImplant->date_fabrication),
                'Pays de fabrication' => [
                    'value' => $tigeImplant->pays_fabrication,
                    'autre' => null,
                ],
            ] : null,
            'Resurfaçage' => $this->formatBool($details->resurfacage),
        ];
    }

    /**
     * Construction de la section PTH de reprise
     */
    private function buildPTHRepriseSection($intervention): ?array
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        if (!$details) return null;

        $detailsSupp = $this->decodeJsonField($details->details_supplementaires, []);

        // Vérifier si c'est bien une reprise
        $typePth = $detailsSupp['pth']['si_pth'] ?? null;
        if ($typePth !== 'reprise' && empty($detailsSupp['composant_acetabulaire_reprise'])) {
            return null;
        }

        // Récupérer les implants de reprise
        $cupuleImplant = $intervention->implants->firstWhere('type_composant', 'Cupule (reprise)');
        $tigeImplant = $intervention->implants->firstWhere('type_composant', 'Tige fémorale (reprise)');

        // Fallback sur les implants normaux si pas de reprise spécifique
        if (!$cupuleImplant) {
            $cupuleImplant = $intervention->implants->firstWhere('type_composant', 'Cupule');
        }
        if (!$tigeImplant) {
            $tigeImplant = $intervention->implants->firstWhere('type_composant', 'Tige fémorale');
        }

        $compAcetab = $detailsSupp['composant_acetabulaire_reprise'] ?? $detailsSupp['composant_acetabulaire'] ?? [];
        $tigeFem = $detailsSupp['tige_femorale_reprise'] ?? $detailsSupp['tige_femorale'] ?? [];
        $synthFem = $detailsSupp['synthese_femorale_reprise'] ?? $detailsSupp['synthese_femorale'] ?? [];

        return [
            'Type de tige' => $detailsSupp['type_tige'] ?? null,
            'Type d\'armature cotyloïdienne' => $detailsSupp['type_armature'] ?? null,
            'Composant acétabulaire' => [
                'Type de fixation' => $compAcetab['type_fixation'] ?? null,
                'Alliage' => $compAcetab['alliage'] ?? null,
                'Revêtement de surface' => $compAcetab['revetement_surface'] ?? null,
                'Reconstruction acétabulaire' => $compAcetab['reconstruction'] ?? null,
                'Taille : cupule' => $compAcetab['taille_cupule'] ?? null,
                'Materiaux Insert' => $compAcetab['materiaux_insert'] ?? null,
            ],
            'Traçabilité (Cupule)' => $cupuleImplant ? [
                'Société' => $cupuleImplant->societe,
                'Marque de la prothèse' => $cupuleImplant->marque_modele,
                'Référence' => $cupuleImplant->reference,
                'Numéro de lot' => $cupuleImplant->numero_lot,
                'UDI-DI / UDI-PI' => $cupuleImplant->udi_di_pi,
                'Date de fabrication' => $this->formatDate($cupuleImplant->date_fabrication),
                'Pays de fabrication' => [
                    'value' => $cupuleImplant->pays_fabrication,
                    'autre' => null,
                ],
            ] : null,
            'Tige fémorale' => [
                'Collerette' => $tigeFem['collerette'] ?? $details->tige_femoral_collerette,
                'Taille' => $tigeFem['taille'] ?? null,
                'Longueur du col' => $tigeFem['longueur_col'] ?? null,
                'Tête' => $tigeFem['tete'] ?? null,
                'Type de fixation' => $tigeFem['type_fixation'] ?? $details->tige_femoral_fixation,
                'Alliage/Tige femorale' => $tigeFem['alliage'] ?? null,
                'Revêtement de surface' => $tigeFem['revetement_surface'] ?? null,
                'Position de la tige' => $tigeFem['position'] ?? null,
            ],
            'Synthèse fémorale complémentaire' => [
                'Type' => $synthFem['type'] ?? null,
                'Drainage' => isset($synthFem['drainage']) ? $this->formatBool($synthFem['drainage']) : null,
            ],
            'Traçabilité (Tige)' => $tigeImplant ? [
                'Société' => $tigeImplant->societe,
                'Marque de la tige' => $tigeImplant->marque_modele,
                'Référence' => $tigeImplant->reference,
                'Numéro de lot' => $tigeImplant->numero_lot,
                'UDI-DI / UDI-PI' => $tigeImplant->udi_di_pi,
                'Date de fabrication' => $this->formatDate($tigeImplant->date_fabrication),
                'Pays de fabrication' => [
                    'value' => $tigeImplant->pays_fabrication,
                    'autre' => null,
                ],
            ] : null,
            'Resurfaçage' => $this->formatBool($details->resurfacage),
        ];
    }

    // =====================================================
    // MÉTHODES SPÉCIFIQUES POUR LE GENOU
    // À ajouter dans le Controller
    // =====================================================

    /**
     * Construction de la section type d'arthroplastie pour le genou
     */
    private function buildKneeArthroplastySection($intervention): ?array
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        if (!$details) return null;

        $detailsSupp = $this->decodeJsonField($details->details_supplementaires, []);

        return [
            'Type' => $details->type_arthroplastie,
            'Si prothèse primaire' => [
                'value' => $details->type_prothese_primaire,
                'autre' => $detailsSupp['type_prothese_primaire_autre'] ?? null,
            ],
        ];
    }

    /**
     * Construction de la section implants utilisés pour le genou
     */
    private function buildKneeImplantsSection($intervention): ?array
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        $detailsSupp = $details ? $this->decodeJsonField($details->details_supplementaires, []) : [];

        // Récupérer les implants par type
        $femoralImplant = $intervention->implants->firstWhere('type_composant', 'Composant fémoral');
        $tibialImplant = $intervention->implants->firstWhere('type_composant', 'Composant tibial');
        $insertImplant = $intervention->implants->firstWhere('type_composant', 'Insert');
        $patellaImplant = $intervention->implants->firstWhere('type_composant', 'Composant patellaire');

        // Détails spécifiques des implants
        $femoralDetails = $femoralImplant ? $this->decodeJsonField($femoralImplant->details_specifiques, []) : [];
        $tibialDetails = $tibialImplant ? $this->decodeJsonField($tibialImplant->details_specifiques, []) : [];
        $insertDetails = $insertImplant ? $this->decodeJsonField($insertImplant->details_specifiques, []) : [];
        $patellaDetails = $patellaImplant ? $this->decodeJsonField($patellaImplant->details_specifiques, []) : [];

        // Données supplémentaires stockées dans details_supplementaires
        $compFemoral = $detailsSupp['composant_femoral'] ?? [];
        $compTibial = $detailsSupp['composant_tibial'] ?? [];
        $compInsert = $detailsSupp['insert'] ?? [];
        $compPatella = $detailsSupp['composant_patellaire'] ?? [];

        return [
            'Composant fémoral' => [
                'Marque / Modèle' => $femoralImplant?->marque_modele ?? $compFemoral['marque_modele'] ?? null,
                'Taille' => $this->formatValueAutre($femoralDetails['taille'] ?? $compFemoral['taille'] ?? null, $femoralDetails['taille_autre'] ?? null),
                'Fixation' => $details?->fixation_femoral ?? $femoralDetails['fixation'] ?? null,
                'Alliage' => $this->formatValueAutre($femoralDetails['alliage'] ?? $compFemoral['alliage'] ?? null, $femoralDetails['alliage_autre'] ?? null),
                'Société (Traçabilité)' => $femoralImplant?->societe,
                'Référence (Traçabilité)' => $femoralImplant?->reference,
                'Numéro de lot (Traçabilité)' => $femoralImplant?->numero_lot,
                'UDI-DI / UDI-PI (Traçabilité)' => $femoralImplant?->udi_di_pi,
                'Date de fabrication (Traçabilité)' => $this->formatDate($femoralImplant?->date_fabrication),
                'Pays de fabrication (Traçabilité)' => [
                    'value' => $femoralImplant?->pays_fabrication,
                    'autre' => null,
                ],
            ],
            'Composant tibial (plateau)' => [
                'Marque / Modèle' => $tibialImplant?->marque_modele ?? $compTibial['marque_modele'] ?? null,
                'Taille' => $this->formatValueAutre($tibialDetails['taille'] ?? $compTibial['taille'] ?? null, $tibialDetails['taille_autre'] ?? null),
                'Fixation' => $details?->fixation_tibial ?? $tibialDetails['fixation'] ?? null,
                'Plateau' => $tibialDetails['plateau'] ?? $compTibial['plateau'] ?? null,
                'Présence de quille' => [
                    'value' => isset($tibialDetails['presence_quille']) ? $this->formatBool($tibialDetails['presence_quille']) : ($compTibial['presence_quille']['value'] ?? null),
                    'autre' => $tibialDetails['quille_longueur_mm'] ?? ($compTibial['presence_quille']['autre'] ?? null),
                ],
                'Alliage' => $this->formatValueAutre($tibialDetails['alliage'] ?? $compTibial['alliage'] ?? null, $tibialDetails['alliage_autre'] ?? null),
            ],
            'Insert (polyéthylène)' => [
                'Matériau' => $this->formatValueAutre($insertDetails['materiau'] ?? $compInsert['materiau'] ?? null, $insertDetails['materiau_autre'] ?? null),
                'Épaisseur (mm)' => $insertDetails['epaisseur_mm'] ?? $compInsert['epaisseur_mm'] ?? null,
                'Mobilité' => $details?->mobilite_insert ?? $insertDetails['mobilite'] ?? null,
                'Design' => $details?->design_insert ?? $insertDetails['design'] ?? null,
                'Société (Traçabilité)' => $insertImplant?->societe,
                'Référence (Traçabilité)' => $insertImplant?->reference,
                'Numéro de lot (Traçabilité)' => $insertImplant?->numero_lot,
                'UDI-DI / UDI-PI (Traçabilité)' => $insertImplant?->udi_di_pi,
                'Date de fabrication (Traçabilité)' => $this->formatDate($insertImplant?->date_fabrication),
                'Pays de fabrication (Traçabilité)' => [
                    'value' => $insertImplant?->pays_fabrication,
                    'autre' => null,
                ],
            ],
            'Composant patellaire' => [
                'Patella posée' => $this->formatBool($details?->patella_posee ?? $patellaDetails['patella_posee'] ?? false),
                'Type' => $this->formatValueAutre($patellaDetails['type'] ?? $compPatella['type'] ?? null, $patellaDetails['type_autre'] ?? null),
                'Fixation' => $patellaDetails['fixation'] ?? $compPatella['fixation'] ?? null,
                'Diamètre (mm)' => $patellaDetails['diametre_mm'] ?? $compPatella['diametre_mm'] ?? null,
                'Réf. / Numéro de lot / UDI-PI' => $patellaImplant?->reference,
            ],
        ];
    }

    /**
     * Construction de la section drainage (genou)
     */
    private function buildDrainageSection($intervention): ?string
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        if (!$details) return null;

        $detailsSupp = $this->decodeJsonField($details->details_supplementaires, []);

        if (isset($detailsSupp['drainage'])) {
            return $this->formatBool($detailsSupp['drainage']);
        }

        return null;
    }

    /**
     * Helper pour formater un champ value/autre
     */
    private function formatValueAutre($value, $autre = null)
    {
        if (is_array($value)) {
            return [
                'value' => $value['value'] ?? null,
                'autre' => $value['autre'] ?? $autre,
            ];
        }

        if ($value && $autre) {
            return [
                'value' => $value,
                'autre' => $autre,
            ];
        }

        return $value;
    }

    /**
     * Met à jour une intervention existante et toutes ses données associées.
     * C'est la méthode UPDATE.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id L'ID de l'intervention à mettre à jour
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        // 1. Validation identique au store
        try {
            $validator = Validator::make($request->all(), [
                'surgeon_firebase_uid' => 'required|string|exists:surgeons,firebase_uid',
                'formType' => 'required|string|in:hanche,genou',
                'IDENTITE DU PATIENT' => 'required|array',
                'IDENTITE DU PATIENT.Nom' => 'required|string|max:255',
                'DONNEES CLINIQUES' => 'required|array',
                'Anesthésie' => 'required|array',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erreur de validation',
                    'errors' => $validator->errors()
                ], 422);
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Erreur de validation: ' . $e->getMessage()], 422);
        }
        
        // 2. Récupération sécurisée
        $surgeon = Surgeon::where('firebase_uid', $request->input('surgeon_firebase_uid'))->first();
        $intervention = Intervention::where('surgeon_id', $surgeon->id)->findOrFail($id);

        DB::beginTransaction();
        try {
            // == 1. Mise à jour du Patient ==
            $patientData = $request->input('IDENTITE DU PATIENT');
            
            // Extraction profession
            $profession = null;
            $professionAutre = null;
            if (isset($patientData['Profession'])) {
                if (is_array($patientData['Profession'])) {
                    $profession = $patientData['Profession']['value'] ?? null;
                    $professionAutre = $patientData['Profession']['autre'] ?? null;
                } else {
                    $profession = $patientData['Profession'];
                }
            }

            $intervention->patient()->update([
                'nom' => $patientData['Nom'],
                'prenom' => $patientData['Prénom(s)'] ?? null,
                'date_naissance' => $this->parseDate($patientData['Date de naissance (JJ-MM-AAAA)'] ?? null),
                'sexe' => $patientData['Sexe'] ?? null,
                'profession' => $profession,
                'profession_autre' => $professionAutre,
                'adresse' => $patientData['Adresse du patient'] ?? null,
                'telephone' => $patientData['Téléphone'] ?? null,
                'numero_identifiant' => $patientData['Numéro Identifiant'] ?? null,
            ]);

            // == 2. Mise à jour de l'Intervention ==
            $interventionData = $request->input('DONNEES THERAPEUTIQUES.Chirurgicales', []);
            $chirurgienData = $request->input('Chirurgien', []);
            $formType = $request->input('formType');

            // Voie d'abord
            $voieAbord = null;
            $voieAbordAutre = null;
            if ($formType === 'hanche') {
                $voieAbordData = $interventionData["Voie d'abord de la hanche"] ?? null;
                if (is_array($voieAbordData)) {
                    $voieAbord = $voieAbordData['value'] ?? null;
                    $voieAbordAutre = $voieAbordData['autre'] ?? null;
                } else {
                    $voieAbord = $voieAbordData;
                }
            } else {
                $voieAbordArray = $interventionData["Voie d'abord"] ?? [];
                $voieAbord = is_array($voieAbordArray) ? json_encode($voieAbordArray) : $voieAbordArray;
            }

            // Hôpital
            $hopitalData = $patientData["Hôpital / numéro d'établissement"] ?? null;
            $hopital = null;
            if (is_array($hopitalData)) {
                $hopital = $hopitalData['value'] ?? $hopitalData['autre'] ?? null;
                if ($hopital === 'autre') {
                    $hopital = $hopitalData['autre'] ?? null;
                }
            } else {
                $hopital = $hopitalData;
            }

            $intervention->update([
                'type_prothese' => $formType,
                'hopital_etablissement' => $hopital,
                'date_intervention' => $this->parseDate($interventionData['Date intervention'] ?? null),
                'heure_debut' => $interventionData['Heure début'] ?? null,
                'heure_fin' => $interventionData['Fin'] ?? null,
                'chirurgien_identifiant' => $chirurgienData['Identifiant/code'] ?? null,
                'voie_dabord' => $voieAbord,
                'voie_dabord_autre' => $voieAbordAutre,
                'donnees_formulaire_autres' => $this->extractAutresDonnees($request, $formType),
            ]);

            // == 3. Mise à jour des relations 1:1 (updateOrCreate) ==
            
            // Antécédents
            $antecedentsData = $request->input('TERRAIN (S) - ANTECEDENTS') ?? $request->input('TERRAIN (S) - ANTECEDANTS');
            if ($antecedentsData) {
                $interventionAnt = null;
                $interventionAntAutre = null;
                $interventionAntData = $antecedentsData['Intervention chirurgicale sur cette hanche'] ?? $antecedentsData['Intervention chirurgicale sur ce genou'] ?? null;
                
                if (is_array($interventionAntData)) {
                    $interventionAnt = $interventionAntData['value'] ?? null;
                    $interventionAntAutre = $interventionAntData['autre'] ?? null;
                } else {
                    $interventionAnt = $interventionAntData;
                }

                $intervention->antecedentsTerrains()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'antecedents_medicaux' => $antecedentsData['Antécédents médicaux'] ?? [],
                        'serologie_retroviral' => $antecedentsData['Sérologie rétroviral'] ?? null,
                        'traumatisme' => $this->parseBoolean($antecedentsData['Traumatisme'] ?? null),
                        'corticotherapie' => $this->parseBoolean($antecedentsData['Corticothérapie'] ?? null),
                        'intervention_anterieure' => $interventionAnt,
                        'intervention_anterieure_autre' => $interventionAntAutre,
                    ]
                );
            }

            // Données Cliniques
            $cliniquesData = $request->input('DONNEES CLINIQUES');
            if ($cliniquesData) {
                $poids = $this->parseNumeric($cliniquesData['Poids (kg)'] ?? null);
                $taille_cm = $this->parseNumeric($cliniquesData['Taille (cm)'] ?? null);
                $imc = $this->parseNumeric($cliniquesData['IMC (kg/m²)'] ?? null);

                if (is_null($imc) && $poids && $taille_cm && $taille_cm > 0) {
                    $taille_m = $taille_cm / 100;
                    $imc = round($poids / ($taille_m * $taille_m), 2);
                }

                $impotence = false;
                $impotencePrecision = null;
                $impotenceData = $cliniquesData['Impotence fonctionnelle'] ?? null;
                if (is_array($impotenceData)) {
                    $impotence = $this->parseBoolean($impotenceData['value'] ?? null);
                    $impotencePrecision = $impotenceData['conditional_value'] ?? null;
                } else {
                    $impotence = $this->parseBoolean($impotenceData);
                }

                $ilmi = false;
                $ilmiCm = null;
                $ilmiData = $cliniquesData['ILMI'] ?? null;
                if (is_array($ilmiData)) {
                    $ilmi = $this->parseBoolean($ilmiData['value'] ?? null);
                    $ilmiCm = $this->parseNumeric($ilmiData['autre'] ?? null);
                } else {
                    $ilmi = $this->parseBoolean($ilmiData);
                }

                $intervention->donneesCliniques()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'poids_kg' => $poids,
                        'taille_cm' => $taille_cm,
                        'imc' => $imc,
                        'cote_opere' => $cliniquesData['Côté'] ?? null,
                        'boiterie' => $this->parseBoolean($cliniquesData['Boiterie'] ?? null),
                        'impotence_fonctionnelle' => $impotence,
                        'impotence_fonctionnelle_precision' => $impotencePrecision,
                        'pma_pre_operatoire' => $cliniquesData['PMA'] ?? null,
                        'score_parker' => $cliniquesData['Score de Parker'] ?? null,
                        'ilmi' => $ilmi,
                        'ilmi_cm' => $ilmiCm,
                    ]
                );
            }

            // Données Paracliniques
            $paracliniquesData = $request->input('DONNEES PARACLINIQUES');
            if ($paracliniquesData) {
                if (isset($paracliniquesData['Biologie'])) {
                    $bioData = $paracliniquesData['Biologie'];
                    $intervention->donneesBiologiques()->updateOrCreate(
                        ['intervention_id' => $intervention->id],
                        [
                            'globules_blancs' => $this->parseNumeric($bioData['Globules Blancs (10³/mm³)'] ?? null),
                            'neutrophiles' => $this->parseNumeric($bioData['Neutrophiles (10³/mm³)'] ?? $bioData['Neutrophiles'] ?? null),
                            'hemoglobine' => $this->parseNumeric($bioData['Hémoglobine (g/dl)'] ?? null),
                            'plaquettes' => $this->parseNumeric($bioData['Plaquettes (10³/mm³)'] ?? null),
                            'groupe_sanguin' => $bioData['Groupe Sanguin'] ?? null,
                            'rhesus' => $bioData['Rhésus'] ?? null,
                            'creatinine' => $this->parseNumeric($bioData['Créatininémie (µmol/L)'] ?? $bioData['Créatinine (mmol/L)'] ?? null),
                            'glycemie' => $this->parseNumeric($bioData['Glycémie (mmol/L)'] ?? null),
                            'crp' => $this->parseNumeric($bioData['CRP (mg/L)'] ?? $bioData['CRP (g/L)'] ?? null),
                            'electrophorese_hemoglobine' => $bioData['Electrophorèse de l\'hémoglobine'] ?? null,
                            'procalcitonine' => $bioData['Procalcitonine (ng/mL)'] ?? $bioData['Procalcitonine'] ?? null,
                            'tp_tck' => $bioData['TP/TCK'] ?? null,
                            'inr' => $bioData['INR'] ?? null,
                            'autre_biologie' => $bioData['Autre (préciser)'] ?? null,
                        ]
                    );
                }

                if (isset($paracliniquesData['Imagerie (Diagnostic)'])) {
                    $imagerieData = $paracliniquesData['Imagerie (Diagnostic)'];
                    $resultatPrincipal = null;
                    $resultatPrincipalAutre = null;
                    $diagnosticData = $imagerieData['Résultats (cocher le principal)'] ?? $imagerieData['Diagnostic (cocher le principal)'] ?? null;
                    
                    if (is_array($diagnosticData)) {
                        $resultatPrincipal = $diagnosticData['value'] ?? null;
                        $resultatPrincipalAutre = $diagnosticData['autre'] ?? null;
                    } else {
                        $resultatPrincipal = $diagnosticData;
                    }

                    $intervention->imageriesDiagnostics()->updateOrCreate(
                        ['intervention_id' => $intervention->id],
                        [
                            'techniques' => json_encode($imagerieData['Techniques'] ?? []),
                            'resultat_principal' => $resultatPrincipal,
                            'resultat_principal_autre' => $resultatPrincipalAutre,
                        ]
                    );
                }
            }

            // Anesthésie
            $anesthesieData = $request->input('Anesthésie');
            if ($anesthesieData) {
                $typeAnesthesie = null;
                $typeAnesthesieAutre = null;
                $typeData = $anesthesieData['Type'] ?? null;
                if (is_array($typeData)) {
                    $typeAnesthesie = $typeData['value'] ?? null;
                    $typeAnesthesieAutre = $typeData['autre'] ?? null;
                } else {
                    $typeAnesthesie = $typeData;
                }

                $complicationsPerAnesth = false;
                $complicationsPerAnesthDetails = null;
                $complicationsData = $anesthesieData['Complications per-anesthésiques'] ?? null;
                if (is_array($complicationsData)) {
                    $complicationsPerAnesth = $this->parseBoolean($complicationsData['value'] ?? null);
                    $complicationsPerAnesthDetails = $complicationsData['autre'] ?? null;
                } else {
                    $complicationsPerAnesth = $this->parseBoolean($complicationsData);
                }

                $intervention->donneesAnesthesiques()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'type_anesthesie' => $typeAnesthesie,
                        'type_anesthesie_autre' => $typeAnesthesieAutre,
                        'classification_asa' => $anesthesieData['Classification ASA'] ?? null,
                        'complications_per_anesthesiques' => $complicationsPerAnesth,
                        'complications_per_anesthesiques_details' => $complicationsPerAnesthDetails,
                    ]
                );
            }

            // Suites Opératoires
            $suitesData = $request->input('Suites opératoires et Résultats');
            if ($suitesData) {
                $priseEnCharge = null;
                $priseEnChargeAutre = null;
                $priseEnChargeData = $suitesData['Prise en charge des frais de soins'] ?? null;
                if (is_array($priseEnChargeData)) {
                    $priseEnCharge = $priseEnChargeData['value'] ?? null;
                    $priseEnChargeAutre = $priseEnChargeData['autre'] ?? null;
                } else {
                    $priseEnCharge = $priseEnChargeData;
                }

                $reeducation = false;
                $reeducationNbSeances = null;
                $reeducationData = $suitesData['Rééducation fonctionnelle'] ?? null;
                if (is_array($reeducationData)) {
                    $reeducation = $this->parseBoolean($reeducationData['value'] ?? null);
                    $reeducationNbSeances = $reeducationData['autre'] ?? null;
                } else {
                    $reeducation = $this->parseBoolean($reeducationData);
                }

                $correctionRaccourcissement = false;
                $correctionRaccourcissementCm = null;
                $correctionRaccData = $suitesData['Correction d\'un raccourcissement'] ?? null;
                if (is_array($correctionRaccData)) {
                    $correctionRaccourcissement = $this->parseBoolean($correctionRaccData['value'] ?? null);
                    $correctionRaccourcissementCm = $this->parseNumeric($correctionRaccData['autre'] ?? null);
                } else {
                    $correctionRaccourcissement = $this->parseBoolean($correctionRaccData);
                }

                $descellementType = null;
                $descellementSiege = null;
                $siDescellement = $suitesData['Si descellement'] ?? null;
                if (is_array($siDescellement)) {
                    $descellementType = $siDescellement['Type'] ?? null;
                    $descellementSiege = $siDescellement['Siège'] ?? null;
                }

                $intervention->suitesOperatoires()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'delai_mise_en_charge_jours' => $this->parseNumeric($suitesData['Délai de mise en charge (jours)'] ?? null),
                        'delai_appui_total_mois' => $this->parseNumeric($suitesData['Délai d\'appui total (mois)'] ?? null),
                        'duree_sejour_hospitalier' => $suitesData['Durée de séjour hospitalier'] ?? null,
                        'reeducation_fonctionnelle' => $reeducation,
                        'reeducation_nb_seances' => $reeducationNbSeances,
                        'complications_per_operatoire' => json_encode($suitesData['Complications per opératoire'] ?? []),
                        'suites_immediates' => $suitesData['Suites opératoires immédiates'] ?? null,
                        'complications_immediates' => json_encode($suitesData['Complications immédiates'] ?? []),
                        'complications_secondaires' => json_encode($suitesData['Complications secondaires'] ?? []),
                        'complications_tardives' => json_encode($suitesData['Complications tardives'] ?? []),
                        'prise_en_charge_frais' => $priseEnCharge,
                        'prise_en_charge_frais_autre' => $priseEnChargeAutre,
                        'dernier_recul_mois' => $suitesData['Dernier recul (mois)'] ?? null,
                        'eva_patient' => $suitesData['Appréciation du patient lui-même : EVA'] ?? null,
                        'correction_raccourcissement' => $correctionRaccourcissement,
                        'correction_raccourcissement_cm' => $correctionRaccourcissementCm,
                        'correction_boiterie' => $this->parseBoolean($suitesData['Correction d\'une boiterie'] ?? null),
                        'pma_post_operatoire' => $suitesData['Evaluation fonctionnelle : PMA'] ?? null,
                        'descellement_type' => $descellementType,
                        'descellement_siege' => $descellementSiege,
                    ]
                );
            }

            // Révision
            $revisionData = $request->input('En cas de révision');
            if ($revisionData) {
                $typesRevisionData = $revisionData['Type de révision'] ?? [];
                $motifsRevisionData = $revisionData['Motif de la révision'] ?? [];
                
                $typesRevision = $typesRevisionData['Type'] ?? [];
                $typesRevisionAutre = null;
                if (is_array($typesRevision) && in_array('Autre', $typesRevision)) {
                    $typesRevisionAutre = $typesRevisionData['Lequel(s) / Autre (préciser)'] ?? null;
                }
                
                $motifsRevision = $motifsRevisionData['Motif'] ?? [];
                $motifsRevisionAutre = null;
                if (is_array($motifsRevision) && in_array('Autre', $motifsRevision)) {
                    $motifsRevisionAutre = $motifsRevisionData['Fracture Spécifiée / Autre'] ?? null;
                }
                
                if (!empty($typesRevision) || !empty($motifsRevision)) {
                    $intervention->revision()->updateOrCreate(
                        ['intervention_id' => $intervention->id],
                        [
                            'est_une_revision' => true,
                            'types_revision' => json_encode($typesRevision),
                            'types_revision_autre' => $typesRevisionAutre,
                            'motifs_revision' => json_encode($motifsRevision),
                            'motifs_revision_autre' => $motifsRevisionAutre,
                        ]
                    );
                }
            }

            // Greffe Osseuse
            $greffeData = $request->input('Greffe osseuse');
            if ($greffeData && $this->parseBoolean($greffeData['Transplantation osseuse'] ?? null)) {
                $localisation = $greffeData['Localisation'] ?? null;
                $intervention->greffeOsseuse()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'type' => $greffeData['Type'] ?? null,
                        'localisation' => is_array($localisation) ? json_encode($localisation) : $localisation,
                    ]
                );
            }

            // Système de Navigation
            $navData = $request->input('Système de navigation');
            if ($navData && $this->parseBoolean($navData['Utilisation'] ?? null)) {
                $intervention->systemeNavigation()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'systeme_utilise' => $navData['Système utilisé'] ?? null,
                        'numero_serie' => $navData['Numéro de série'] ?? null,
                        'chirurgie_mini_invasive' => $this->parseBoolean($navData['Chirurgie mini-invasive'] ?? null),
                    ]
                );
            }

            // Analgésie
            $analgesieData = $request->input('Analgésie par infiltration locale ou loco-régionale');
            if ($analgesieData) {
                $intervention->analgesie()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'infiltration' => $this->parseBoolean($analgesieData['Infiltration'] ?? null),
                        'catheter_en_place' => $this->parseBoolean($analgesieData['Cathéter laissé en place'] ?? null),
                        'produit' => $analgesieData['Produit'] ?? null,
                    ]
                );
            }

            // Prophylaxie
            $prophylaxieData = $request->input('Prophylaxie antibiotique');
            if ($prophylaxieData && $this->parseBoolean($prophylaxieData['Utilisée'] ?? null)) {
                $intervention->prophylaxieAntibiotique()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'antibiotique' => $prophylaxieData['Si oui préciser antibiotique'] ?? null,
                        'dose_par_jour' => $prophylaxieData['Dose /jour'] ?? null,
                        'debute_per_operatoire' => $this->parseBoolean($prophylaxieData['Débuté en per-opératoire'] ?? null),
                        'duree' => $prophylaxieData['Durée'] ?? null,
                    ]
                );
            }

            // Difficultés Opératoires
            $difficulteData = $request->input('Difficulté opératoire');
            if ($difficulteData) {
                $surComposante = $difficulteData['Sur'] ?? null;
                if (is_array($surComposante)) {
                    $surComposante = json_encode($surComposante);
                }
                
                $nature = [];
                $natureAutre = null;
                $natureData = $difficulteData['Nature'] ?? null;
                if (is_array($natureData)) {
                    if (isset($natureData['value'])) {
                        $nature = $natureData['value'];
                        $natureAutre = $natureData['autre'] ?? null;
                    } else {
                        $nature = $natureData;
                    }
                }

                $intervention->difficulteOperatoire()->updateOrCreate(
                    ['intervention_id' => $intervention->id],
                    [
                        'sur_composante' => $surComposante,
                        'nature' => json_encode($nature),
                        'nature_autre' => $natureAutre,
                    ]
                );
            }

            // == 4. Mise à jour des Implants (DELETE + RECREATE) ==
            $intervention->implants()->delete();
            $this->createImplants($intervention, $request, $formType);

            // == 5. Mise à jour des Détails Thérapeutiques ==
            $this->createDetailsTherapeutiques($intervention, $request, $formType);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Intervention mise à jour avec succès.',
                'data' => [
                    'intervention_id' => $intervention->id,
                    'patient_id' => $intervention->patient_id,
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erreur update formulaire: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Une erreur est survenue lors de la mise à jour.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Supprime une intervention et toutes ses données associées,
     * après avoir vérifié que l'action est autorisée.
     * C'est la méthode DELETE.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id L'ID de l'intervention à supprimer
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request, $id)
    {
        // On valide que l'UID du chirurgien a bien été fourni dans le corps de la requête.
        $request->validate(['surgeon_firebase_uid' => 'required|string|exists:surgeons,firebase_uid']);

        // On récupère le chirurgien qui effectue la demande.
        $surgeon = Surgeon::where('firebase_uid', $request->input('surgeon_firebase_uid'))->first();

        // REQUÊTE SÉCURISÉE : On cherche l'intervention à supprimer
        // uniquement parmi celles qui appartiennent au chirurgien authentifié.
        $intervention = Intervention::where('surgeon_id', $surgeon->id)->findOrFail($id);
        // Si l'intervention n'est pas trouvée ou n'appartient pas au chirurgien,
        // findOrFail renverra automatiquement une erreur 404 (Not Found), empêchant toute action non autorisée.

        // ACTION DE SUPPRESSION CORRECTE : On supprime l'intervention.
        // Grâce à `onDelete('cascade')` dans nos migrations, toutes les données dans les tables
        // 'antecedents_terrains', 'donnees_cliniques', 'implants', etc. liées à cette intervention
        // seront automatiquement supprimées par la base de données.
        // Le patient, lui, ne sera PAS supprimé.
        $intervention->delete();

        // On retourne une réponse HTTP 204 No Content, qui est la norme pour
        // une suppression réussie qui n'a pas besoin de renvoyer de contenu.
        return response()->json(null, 204);
    }
}