
---

### **1. Fichier de Routes (`routes/api.php`)**

C'est le cœur de la déclaration de votre API. Grâce à la puissance de Laravel, toutes les routes nécessaires pour un CRUD complet sont créées par une seule ligne de code.

Ouvrez le fichier `routes/api.php` et assurez-vous qu'il contient ceci :

```php
<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\FormulaireController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
*/

// Cette seule ligne de code déclare toutes les routes CRUD pour vos formulaires.
// Elle lie les URLs aux méthodes correspondantes dans votre FormulaireController.
Route::apiResource('formulaires', FormulaireController::class);
```

---

### **2. Documentation Détaillée des Endpoints**

Voici la description de chaque endpoint généré par `apiResource`.

**Concept Clé : Authentification**

Toutes les requêtes doivent inclure l'identifiant unique (UID) du chirurgien fourni par Firebase. C'est ce qui garantit la sécurité et le cloisonnement des données. Le nom de ce paramètre est `surgeon_firebase_uid`.

---

#### **Endpoint 1 : Lister les interventions d'un chirurgien**

-   **Action :** Récupère une liste paginée de toutes les interventions appartenant au chirurgien spécifié.
-   **Méthode HTTP :** `GET`
-   **Endpoint :** `/api/formulaires`
-   **Paramètres Requis :**
    -   `surgeon_firebase_uid` (string, dans la query string) : L'UID Firebase du chirurgien.
-   **Exemple d'Appel (avec axios) :**
    ```javascript
    const firebase_uid = "abc123xyz456"; // UID du chirurgien connecté
    axios.get(`/api/formulaires?surgeon_firebase_uid=${firebase_uid}`);
    ```
-   **Réponse en cas de Succès (Code `200 OK`) :**
    Un objet JSON paginé.
    ```json
    {
        "current_page": 1,
        "data": [
            {
                "id": 42,
                "patient_id": 58,
                "type_prothese": "hanche",
                "date_intervention": "2023-10-27",
                "patient": {
                    "id": 58,
                    "nom": "Dupont",
                    "prenom": "Marie"
                }
            },
            {
                "id": 35,
                "patient_id": 51,
                "type_prothese": "genou",
                "date_intervention": "2023-09-15",
                "patient": {
                    "id": 51,
                    "nom": "Martin",
                    "prenom": "Paul"
                }
            }
        ],
        "first_page_url": "/api/formulaires?page=1",
        "next_page_url": "/api/formulaires?page=2",
        "path": "/api/formulaires",
        "per_page": 15,
        "total": 30
    }
    ```
-   **Réponses en cas d'Erreur :**
    -   `422 Unprocessable Entity` : Si `surgeon_firebase_uid` est manquant ou n'existe pas.

---

#### **Endpoint 2 : Créer une nouvelle intervention**

-   **Action :** Enregistre un nouveau formulaire complet dans la base de données.
-   **Méthode HTTP :** `POST`
-   **Endpoint :** `/api/formulaires`
-   **Paramètres Requis :**
    -   L'objet JSON complet du formulaire (le "payload") dans le corps de la requête.
    -   `surgeon_firebase_uid` (string) doit être inclus à la racine de cet objet JSON.
-   **Exemple d'Appel (avec axios) :**
    ```javascript
    const formData = {
        surgeon_firebase_uid: "abc123xyz456",
        formType: "hanche",
        "IDENTITE DU PATIENT": { /* ... */ },
        "TERRAIN (S) - ANTECEDENTS": { /* ... */ },
        // ... toutes les autres sections
    };
    axios.post('/api/formulaires', formData);
    ```
-   **Réponse en cas de Succès (Code `201 Created`) :**
    ```json
    {
        "message": "Formulaire enregistré avec succès !",
        "intervention_id": 43
    }
    ```
-   **Réponses en cas d'Erreur :**
    -   `422 Unprocessable Entity` : Si des données requises sont manquantes.
    -   `500 Internal Server Error` : Si une erreur survient lors de la transaction.

---

#### **Endpoint 3 : Afficher une intervention spécifique**

-   **Action :** Récupère toutes les données d'une seule intervention et les formate pour remplir le formulaire en mode édition.
-   **Méthode HTTP :** `GET`
-   **Endpoint :** `/api/formulaires/{id}`
-   **Paramètres Requis :**
    -   `id` (integer, dans l'URL) : L'ID de l'intervention à afficher.
    -   `surgeon_firebase_uid` (string, dans la query string) : L'UID Firebase du chirurgien (pour vérification de propriété).
-   **Exemple d'Appel (avec axios) :**
    ```javascript
    const firebase_uid = "abc123xyz456";
    const interventionId = 42;
    axios.get(`/api/formulaires/${interventionId}?surgeon_firebase_uid=${firebase_uid}`);
    ```
-   **Réponse en cas de Succès (Code `200 OK`) :**
    L'objet JSON complet, reconstruit pour correspondre à la structure du formulaire.
    ```json
    {
        "id": 42,
        "formType": "hanche",
        "IDENTITE DU PATIENT": { /* ... */ },
        "TERRAIN (S) - ANTECEDENTS": { /* ... */ },
        // ... toutes les autres sections
    }
    ```
-   **Réponses en cas d'Erreur :**
    -   `404 Not Found` : Si l'ID n'existe pas OU s'il n'appartient pas au chirurgien spécifié.
    -   `422 Unprocessable Entity` : Si `surgeon_firebase_uid` est manquant.

---

#### **Endpoint 4 : Mettre à jour une intervention**

-   **Action :** Met à jour une intervention existante avec de nouvelles données.
-   **Méthode HTTP :** `PUT` (ou `PATCH`)
-   **Endpoint :** `/api/formulaires/{id}`
-   **Paramètres Requis :**
    -   `id` (integer, dans l'URL) : L'ID de l'intervention à modifier.
    -   L'objet JSON complet du formulaire (le "payload") dans le corps de la requête.
    -   `surgeon_firebase_uid` (string) doit être inclus à la racine de cet objet JSON.
-   **Exemple d'Appel (avec axios) :**
    ```javascript
    const interventionId = 42;
    const updatedFormData = {
        surgeon_firebase_uid: "abc123xyz456",
        formType: "hanche",
        "IDENTITE DU PATIENT": { /* nouvelles données ... */ },
        // ... toutes les autres sections
    };
    axios.put(`/api/formulaires/${interventionId}`, updatedFormData);
    ```
-   **Réponse en cas de Succès (Code `200 OK`) :**
    ```json
    {
        "message": "Formulaire mis à jour avec succès !",
        "intervention_id": 42
    }
    ```
-   **Réponses en cas d'Erreur :**
    -   `404 Not Found` : Si l'ID n'existe pas OU s'il n'appartient pas au chirurgien.
    -   `500 Internal Server Error` : Si une erreur survient lors de la transaction de mise à jour.

---

#### **Endpoint 5 : Supprimer une intervention**

-   **Action :** Supprime définitivement une intervention et toutes ses données associées.
-   **Méthode HTTP :** `DELETE`
-   **Endpoint :** `/api/formulaires/{id}`
-   **Paramètres Requis :**
    -   `id` (integer, dans l'URL) : L'ID de l'intervention à supprimer.
    -   `surgeon_firebase_uid` (string) : **Dans le corps (`body`) de la requête DELETE.**
-   **Exemple d'Appel (avec axios) :**
    ```javascript
    const firebase_uid = "abc123xyz456";
    const interventionId = 42;
    axios.delete(`/api/formulaires/${interventionId}`, {
      data: {
        surgeon_firebase_uid: firebase_uid
      }
    });
    ```
-   **Réponse en cas de Succès (Code `204 No Content`) :**
    Aucun contenu n'est retourné, ce qui est la norme pour une suppression réussie.
-   **Réponses en cas d'Erreur :**
    -   `404 Not Found` : Si l'ID n'existe pas OU s'il n'appartient pas au chirurgien.
    -   `422 Unprocessable Entity` : Si `surgeon_firebase_uid` est manquant dans le corps de la requête.


#### **Endpoint 6 : Lister les patients d'un chirurgien**

-   **Action :** Récupère une liste paginée de tous les patients uniques ayant eu au moins une intervention avec le chirurgien spécifié.
-   **Méthode HTTP :** `GET`
-   **Endpoint :** `/api/patients`
-   **Paramètres Requis :**
    -   `surgeon_firebase_uid` (string, dans la query string) : L'UID Firebase du chirurgien.
-   **Exemple d'Appel (avec axios) :**
    ```javascript
    const firebase_uid = "abc123xyz456"; // UID du chirurgien connecté
    axios.get(`/api/patients?surgeon_firebase_uid=${firebase_uid}`);
    ```
-   **Réponse en cas de Succès (Code `200 OK`) :**
    Un objet JSON paginé contenant la liste des patients.
    ```json
    {
        "current_page": 1,
        "data": [
            {
                "id": 58,
                "nom": "Dupont",
                "prenom": "Marie",
                "date_naissance": "1985-06-20",
                "sexe": "Féminin",
                "profession": "Fonctionnaire",
                "profession_autre": null,
                "adresse": "12 rue de la Paix",
                "telephone": "0612345678",
                "created_at": "2023-10-27T10:00:00.000000Z",
                "updated_at": "2023-10-27T10:00:00.000000Z"
            },
            {
                "id": 51,
                "nom": "Martin",
                "prenom": "Paul",
                "date_naissance": "1970-02-10",
                "sexe": "Masculin",
                "profession": null,
                "profession_autre": null,
                "adresse": "45 avenue de la République",
                "telephone": "0798765432",
                "created_at": "2023-09-15T08:30:00.000000Z",
                "updated_at": "2023-09-15T08:30:00.000000Z"
            }
        ],
        "first_page_url": "/api/patients?page=1",
        "next_page_url": null,
        "path": "/api/patients",
        "per_page": 15,
        "total": 2
    }
    ```
-   **Réponses en cas d'Erreur :**
    -   `422 Unprocessable Entity` : Si `surgeon_firebase_uid` est manquant ou n'existe pas.