<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// --- LISTE CORRECTE BASÉE SUR VOS FICHIERS ---
use App\Models\Analgesies;
use App\Models\AntecedentsTerrain;
use App\Models\DetailsTherapeutiquesImplants;
use App\Models\DifficultesOperatoires;
use App\Models\DonneesAnesthesique;
use App\Models\DonneesBiologique;
use App\Models\DonneesClinique;
use App\Models\GreffesOsseuses;
use App\Models\ImageriesDiagnostic;
use App\Models\Implant;
use App\Models\Intervention;
use App\Models\Patient;
use App\Models\ProphylaxiesAntibiotiques;
use App\Models\Revision;
use App\Models\SuitesOperatoire;
use App\Models\Surgeon;
use App\Models\SystemesNavigation;

class FormulaireController extends Controller
{
    /**
     * Affiche une liste paginée des interventions POUR UN CHIRURGIEN SPÉCIFIQUE.
     * C'est la méthode READ (pour une liste).
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        // 1. VALIDATION : L'UID du chirurgien est obligatoire.
        $validated = $request->validate([
            'surgeon_firebase_uid' => 'required|string'
        ]);

        // 2. CRÉATION OU RÉCUPÉRATION DU CHIRURGIEN (LOGIQUE CONSERVÉE)
        // C'est un excellent pattern pour votre application.
        $surgeon = Surgeon::firstOrCreate(
            ['firebase_uid' => $validated['surgeon_firebase_uid']],
            // Le nom peut être fourni optionnellement par le frontend lors de la première connexion
            ['nom' => $request->input('surgeon_name', 'N/A')]
        );

        // 3. REQUÊTE SCOPÉE ET OPTIMISÉE
        $interventions = Intervention::query()
            // SÉCURITÉ : On ne prend que les interventions de ce chirurgien.
            ->where('surgeon_id', $surgeon->id)
            
            // OPTIMISATION : On charge uniquement les colonnes utiles du patient pour la liste.
            // Cela réduit la taille de la réponse et améliore la vitesse.
            ->with('patient:id,nom,prenom,date_naissance') // On ajoute la date de naissance, souvent utile dans les listes.
            
            // ERGONOMIE : On trie par la date d'intervention la plus récente.
            ->latest('date_intervention')
            
            // PERFORMANCE : On pagine les résultats.
            ->paginate(15); // Vous pouvez ajuster le nombre

        // 4. RÉPONSE JSON
        return response()->json($interventions);
    }

    /**
     * Enregistre un nouveau formulaire complet dans la base de données.
     * C'est la méthode CREATE.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        // On valide que l'UID du chirurgien est bien présent dans la requête.
        $request->validate(['surgeon_firebase_uid' => 'required|string',
            'formType' => 'required|string|in:hanche,genou',
            'IDENTITE DU PATIENT' => 'required|array',
            'IDENTITE DU PATIENT.Nom' => 'required|string|max:255',]);

        // La transaction garantit la cohérence des données : tout est enregistré, ou rien ne l'est.
        DB::beginTransaction();

        try {
            // On trouve le chirurgien par son UID Firebase ou on le crée s'il n'existe pas.
            // C'est très pratique pour le premier enregistrement d'un nouveau chirurgien.
            $surgeon = Surgeon::firstOrCreate(
                ['firebase_uid' => $request->input('surgeon_firebase_uid')],
                ['nom' => $request->input('surgeon_name', 'N/A')]
            );

            // == 1. Création ou Récupération du Patient ==
            $patientData = $request->input('IDENTITE DU PATIENT');

            $criteresUnicite = [];
            if (!empty($patientData['Numéro Identifiant'])) {
                $criteresUnicite['numero_identifiant'] = $patientData['Numéro Identifiant'];
            } else {
                $criteresUnicite = [
                    'nom' => $patientData['Nom'],
                    'prenom' => $patientData['Prénom(s)'] ?? null,
                    'date_naissance' => $patientData['Date de naissance (JJ-MM-AAAA)'] ?? null,
                ];
            }

            $donneesPatient = [
                'nom' => $patientData['Nom'],
                'prenom' => $patientData['Prénom(s)'] ?? null,
                'date_naissance' => $patientData['Date de naissance (JJ-MM-AAAA)'] ?? null,
                'sexe' => $patientData['Sexe'] ?? null,
                'profession' => $patientData['Profession']['value'] ?? null,
                'profession_autre' => $patientData['Profession']['autre'] ?? null,
                'adresse' => $patientData['Adresse du patient'] ?? null,
                'telephone' => $patientData['Téléphone'] ?? null,
                'numero_identifiant' => $patientData['Numéro Identifiant'] ?? null,
            ];

            $patient = Patient::updateOrCreate($criteresUnicite, $donneesPatient);

            // == 2. Création de l'Intervention ==
            $interventionData = $request->input('DONNEES THERAPEUTIQUES.Chirurgicales', []);
            $chirurgienData = $request->input('Chirurgien', []);

            // Gestion spéciale pour voie d'abord (peut être array pour genou, string pour hanche)
            $voieAbord = null;
            $voieAbordAutre = null;
            
            if ($request->input('formType') === 'hanche') {
                $voieAbord = $interventionData["Voie d'abord de la hanche"]['value'] ?? null;
                $voieAbordAutre = $interventionData["Voie d'abord de la hanche"]['autre'] ?? null;
            } else {
                // Pour le genou, c'est un array de checkboxes
                $voieAbordArray = $interventionData["Voie d'abord"] ?? [];
                $voieAbord = is_array($voieAbordArray) ? json_encode($voieAbordArray) : $voieAbordArray;
            }

            $intervention = $patient->interventions()->create([
                'surgeon_id' => $surgeon->id,
                'type_prothese' => $request->input('formType'),
                'hopital_etablissement' => $patientData['Hôpital / numéro d\'établissement'] ?? null,
                'date_intervention' => $interventionData['Date intervention'] ?? null,
                'heure_debut' => $interventionData['Heure début'] ?? null,
                'heure_fin' => $interventionData['Fin'] ?? null,
                'chirurgien_identifiant' => $chirurgienData['Identifiant/code'] ?? null,
                'voie_dabord' => $voieAbord,
                'voie_dabord_autre' => $voieAbordAutre,
                'donnees_formulaire_autres' => \Illuminate\Support\Arr::except($request->all(), [
                    'surgeon_firebase_uid','surgeon_name','formType','IDENTITE DU PATIENT',
                    'TERRAIN (S) - ANTECEDENTS','TERRAIN (S) - ANTECEDANTS','DONNEES CLINIQUES','DONNEES PARACLINIQUES',
                    'Chirurgien','Anesthésie','implants_traçabilité','Suites opératoires et Résultats',
                    'En cas de révision','Greffe osseuse','Système de navigation','Analgésie par infiltration locale ou loco-régionale',
                    'Prophylaxie antibiotique','Difficulté opératoire','Implants utilisés','DONNEES THERAPEUTIQUES'
                ])
            ]);

            // == 3. Création des Antécédents ==
            // CORRECTION : Le nom de la section est différent pour le genou
            $antecedentsData = $request->input('TERRAIN (S) - ANTECEDENTS') ?? $request->input('TERRAIN (S) - ANTECEDANTS');
            if ($antecedentsData) {
            $intervention->antecedentsTerrains()->create([
                'antecedents_medicaux' => $antecedentsData['Antécédents médicaux'] ?? [],
                'serologie_retroviral' => $antecedentsData['Sérologie rétroviral'] ?? null,
                'traumatisme' => ($antecedentsData['Traumatisme'] ?? 'Non') === 'Oui',
                'corticotherapie' => ($antecedentsData['Corticothérapie'] ?? 'Non') === 'Oui',
                'intervention_anterieure' => $antecedentsData['Intervention chirurgicale sur cette hanche']['value'] ?? $antecedentsData['Intervention chirurgicale sur ce genou']['value'] ?? null,
                'intervention_anterieure_autre' => $antecedentsData['Intervention chirurgicale sur cette hanche']['autre'] ?? $antecedentsData['Intervention chirurgicale sur ce genou']['autre'] ?? null,
            ]);
            }

            // == 4. Création des Données Cliniques ==
            $cliniquesData = $request->input('DONNEES CLINIQUES');
            $intervention->donneesCliniques()->create([
                'poids_kg' => $cliniquesData['Poids (kg)'],
                'taille_cm' => $cliniquesData['Taille (cm)'],
                'imc' => $cliniquesData['IMC (kg/m²)'],
                'cote_opere' => $cliniquesData['Côté'],
                'boiterie' => ($cliniquesData['Boiterie'] ?? 'Non') === 'Oui',
                'impotence_fonctionnelle' => ($cliniquesData['Impotence fonctionnelle']['value'] ?? 'Non') === 'Oui',
                'impotence_fonctionnelle_precision' => $cliniquesData['Impotence fonctionnelle']['conditional_value'] ?? null,
                'pma_pre_operatoire' => $cliniquesData['PMA'],
                'score_parker' => $cliniquesData['Score de Parker'],
                'ilmi' => ($cliniquesData['ILMI']['value'] ?? 'Non') === 'Oui',
                'ilmi_cm' => $cliniquesData['ILMI']['autre'] ?? null,
            ]);
            
            if ($paracliniquesData = $request->input('DONNEES PARACLINIQUES')) {
                
                // == 5. Création des Données Biologiques ==
                if (isset($paracliniquesData['Biologie'])) {
                    $bioData = $paracliniquesData['Biologie'];
                    $intervention->donneesBiologiques()->create([
                        'globules_blancs' => $bioData['Globules Blancs (10³/mm³)'] ?? null,
                        'neutrophiles' => $bioData['Neutrophiles'] ?? null,
                        'hemoglobine' => $bioData['Hémoglobine (g/dl)'] ?? null,
                        'plaquettes' => $bioData['Plaquettes (10³/mm³)'] ?? null,
                        'groupe_sanguin' => $bioData['Groupe Sanguin'] ?? null,
                        'rhesus' => $bioData['Rhésus'] ?? null,
                        'creatinine' => $bioData['Créatinine (mmol/L)'] ?? null,
                        'glycemie' => $bioData['Glycémie (mmol/L)'] ?? null,
                        'crp' => $bioData['CRP (g/L)'] ?? null,
                        'electrophorese_hemoglobine' => $bioData['Electrophorèse de l\'hémoglobine'] ?? null,
                        'procalcitonine' => $bioData['Procalcitonine'] ?? null,
                        'tp_tck' => $bioData['TP/TCK'] ?? null,
                        'inr' => $bioData['INR'] ?? null,
                        'autre_biologie' => $bioData['Autre (préciser)'] ?? null,
                    ]);
                }
                
                // == 6. Création des Données d'Imagerie ==
                if (isset($paracliniquesData['Imagerie (Diagnostic)'])) {
                    $imagerieData = $paracliniquesData['Imagerie (Diagnostic)'];
                    
                    $intervention->imageriesDiagnostics()->create([
                        'techniques' => json_encode($imagerieData['Techniques'] ?? []),
                        'resultat_principal' => $imagerieData['Résultats (cocher le principal)']['value'] ?? $imagerieData['Diagnostic (cocher le principal)']['value'] ?? null,
                        'resultat_principal_autre' => $imagerieData['Résultats (cocher le principal)']['autre'] ?? $imagerieData['Diagnostic (cocher le principal)']['autre'] ?? null,
                    ]);
                }
            }

            // == 7. Création des Données d'Anesthésie ==
            $anesthesieData = $request->input('Anesthésie');
            $intervention->donneesAnesthesiques()->create([
                'type_anesthesie' => $anesthesieData['Type']['value'] ?? null,
                'type_anesthesie_autre' => $anesthesieData['Type']['autre'] ?? null,
                'classification_asa' => $anesthesieData['Classification ASA'],
                'complications_per_anesthesiques' => ($anesthesieData['Complications per-anesthésiques']['value'] ?? 'Non') === 'Oui',
                'complications_per_anesthesiques_details' => $anesthesieData['Complications per-anesthésiques']['autre'] ?? null,
            ]);

            // == 8. Création des Suites Opératoires ==
            $suitesData = $request->input('Suites opératoires et Résultats');
            if ($suitesData) {
                $intervention->suitesOperatoires()->create([
                    'delai_mise_en_charge_jours' => $suitesData['Délai de mise en charge (jours)'] ?? null,
                    'delai_appui_total_mois' => $suitesData['Délai d\'appui total (mois)'] ?? null,
                    'duree_sejour_hospitalier' => $suitesData['Durée de séjour hospitalier'] ?? null,
                    'complications_per_operatoire' => json_encode($suitesData['Complications per opératoire'] ?? []),
                    'suites_immediates' => $suitesData['Suites opératoires immédiates'] ?? null,
                    'complications_immediates' => json_encode($suitesData['Complications immédiates'] ?? []),
                    'complications_secondaires' => json_encode($suitesData['Complications secondaires'] ?? []),
                    'complications_tardives' => json_encode($suitesData['Complications tardives'] ?? []),
                    'prise_en_charge_frais' => $suitesData['Prise en charge des frais de soins']['value'] ?? null,
                    'prise_en_charge_frais_autre' => $suitesData['Prise en charge des frais de soins']['autre'] ?? null,
                    'dernier_recul_mois' => $suitesData['Dernier recul (mois)'] ?? null,
                    'eva_patient' => $suitesData['Appréciation du patient lui-même : EVA'] ?? null,
                    'pma_post_operatoire' => $suitesData['Evaluation fonctionnelle : PMA'] ?? null,
                ]);
            }

            // == 9. Création des données de Révision ==
            $revisionData = $request->input('En cas de révision');
            if ($revisionData) {
                $intervention->revision()->create([
                    'est_une_revision' => true,
                    'types_revision' => json_encode($revisionData['Type de révision']['Type'] ?? []),
                    'motifs_revision' => json_encode($revisionData['Motif de la révision']['Motif'] ?? []),
                ]);
            }

            // == 10. Création des Données d'Implants ==
            if ($request->input('formType') === 'hanche') {
                $sectionsTracabilite = [
                    'Cupule' => 'PTH (Arthroplastie totale) 1ère intention.Traçabilité (Cupule)',
                    'Tige' => 'PTH (Arthroplastie totale) 1ère intention.Traçabilité (Tige)',
                ];
                
                foreach ($sectionsTracabilite as $typeComposant => $path) {
                    $data = $request->input($path);
                    if ($data) {
                        $intervention->implants()->create([
                            'type_composant' => $typeComposant,
                            'societe' => $data['Société'] ?? null,
                            'marque_modele' => $data['Marque de la prothèse'] ?? $data['Marque de la tige'] ?? null,
                            'reference' => $data['Référence'] ?? null,
                            'numero_lot' => $data['Numéro de lot'] ?? null,
                            'udi_di_pi' => $data['UDI-DI / UDI-PI'] ?? null,
                            'date_fabrication' => $data['Date de fabrication'] ?? null,
                            'pays_fabrication' => $data['Pays de fabrication'] ?? null,
                        ]);
                    }
                }
            } else {
                // Pour le genou - gestion des composants séparément
                $implantsData = $request->input('Implants utilisés', []);
                
                // Composant fémoral
                if (isset($implantsData['Composant fémoral'])) {
                    $data = $implantsData['Composant fémoral'];
                    $intervention->implants()->create([
                        'type_composant' => 'Composant fémoral',
                        'societe' => $data['Société (Traçabilité)'] ?? null,
                        'marque_modele' => $data['Marque / Modèle'] ?? null,
                        'reference' => $data['Référence (Traçabilité)'] ?? null,
                        'numero_lot' => $data['Numéro de lot (Traçabilité)'] ?? null,
                        'udi_di_pi' => $data['UDI-DI / UDI-PI (Traçabilité)'] ?? null,
                        'date_fabrication' => $data['Date de fabrication (Traçabilité)'] ?? null,
                        'pays_fabrication' => $data['Pays de fabrication (Traçabilité)'] ?? null,
                    ]);
                }
                
                // Insert (polyéthylène)
                if (isset($implantsData['Insert (polyéthylène)'])) {
                    $data = $implantsData['Insert (polyéthylène)'];
                    $intervention->implants()->create([
                        'type_composant' => 'Insert',
                        'societe' => $data['Société (Traçabilité)'] ?? null,
                        'marque_modele' => null, // Pas de marque pour l'insert
                        'reference' => $data['Référence (Traçabilité)'] ?? null,
                        'numero_lot' => $data['Numéro de lot (Traçabilité)'] ?? null,
                        'udi_di_pi' => $data['UDI-DI / UDI-PI (Traçabilité)'] ?? null,
                        'date_fabrication' => $data['Date de fabrication (Traçabilité)'] ?? null,
                        'pays_fabrication' => $data['Pays de fabrication (Traçabilité)'] ?? null,
                    ]);
                }
            }

            // == 11. Création des Détails Thérapeutiques ==
            if ($request->input('formType') === 'hanche') {
                $pthData = $request->input('PTH (Arthroplastie totale) 1ère intention');
                if ($pthData) {
                    $intervention->detailsTherapeutiquesImplants()->create([
                        'couple_frottement' => is_array($pthData['Couple de frottement'] ?? null) 
                            ? ($pthData['Couple de frottement'][0] ?? null) 
                            : ($pthData['Couple de frottement'] ?? null),
                        'type_cupule' => $pthData['Composant acétabulaire']['Type de cupule'] ?? null,
                        'tige_femoral_collerette' => $pthData['Tige fémorale']['Collerette'] ?? null,
                        'tige_femoral_fixation' => $pthData['Tige fémorale']['Type de fixation'] ?? null,
                        'resurfacage' => is_array($pthData['Resurfaçage'] ?? null) 
                            ? (($pthData['Resurfaçage'][0] ?? 'Non') === 'Oui')
                            : (($pthData['Resurfaçage'] ?? 'Non') === 'Oui'),
                    ]);
                }
            } elseif ($request->input('formType') === 'genou') {
                $typeArthroplastieData = $request->input('DONNEES THERAPEUTIQUES.Type d\'arthroplastie');
                $implantsGenouData = $request->input('Implants utilisés');
                if ($typeArthroplastieData && $implantsGenouData) {
                    $intervention->detailsTherapeutiquesImplants()->create([
                        'type_arthroplastie' => $typeArthroplastieData['Type'] ?? null,
                        'type_prothese_primaire' => $typeArthroplastieData['Si prothèse primaire']['value'] ?? null,
                        'fixation_femoral' => $implantsGenouData['Composant fémoral']['Fixation'] ?? null,
                        'fixation_tibial' => $implantsGenouData['Composant tibial (plateau)']['Fixation'] ?? null,
                        'mobilite_insert' => $implantsGenouData['Insert (polyéthylène)']['Mobilité'] ?? null,
                        'design_insert' => $implantsGenouData['Insert (polyéthylène)']['Design'] ?? null,
                        'patella_posee' => ($implantsGenouData['Composant patellaire']['Patella posée'] ?? 'Non') === 'Oui',
                    ]);
                }
            }

            // == 12. Greffe Osseuse ==
            if ($greffeData = $request->input('Greffe osseuse')) {
                if (($greffeData['Transplantation osseuse'] ?? 'Non') === 'Oui') {
                    $intervention->greffeOsseuse()->create([
                        'type' => $greffeData['Type'] ?? null,
                        'localisation' => is_array($greffeData['Localisation'] ?? null) 
                            ? json_encode($greffeData['Localisation']) 
                            : $greffeData['Localisation'] ?? null,
                    ]);
                }
            }
            
            // == 13. Système de Navigation ==
            if ($navData = $request->input('Système de navigation')) {
                if (($navData['Utilisation'] ?? 'Non') === 'Oui') {
                    $intervention->systemeNavigation()->create([
                        'systeme_utilise' => $navData['Système utilisé'] ?? null,
                        'numero_serie' => $navData['Numéro de série'] ?? null,
                    ]);
                }
            }

            // == 14. Analgésie ==
            if ($analgesieData = $request->input('Analgésie par infiltration locale ou loco-régionale')) {
                $intervention->analgesie()->create([
                    'infiltration' => ($analgesieData['Infiltration'] ?? 'Non') === 'Oui',
                    'catheter_en_place' => ($analgesieData['Cathéter laissé en place'] ?? 'Non') === 'Oui',
                    'produit' => $analgesieData['Produit'] ?? null,
                ]);
            }

            // == 15. Prophylaxie Antibiotique ==
            if ($prophylaxieData = $request->input('Prophylaxie antibiotique')) {
                if (($prophylaxieData['Utilisée'] ?? 'Non') === 'Oui') {
                    $intervention->prophylaxieAntibiotique()->create([
                        'antibiotique' => $prophylaxieData['Si oui préciser antibiotique'] ?? null,
                        'dose_par_jour' => $prophylaxieData['Dose /jour'] ?? null,
                        'debute_per_operatoire' => ($prophylaxieData['Débuté en per-opératoire'] ?? 'Non') === 'Oui',
                        'duree' => $prophylaxieData['Durée'] ?? null,
                    ]);
                }
            }

            // == 16. Difficultés Opératoires ==
            if ($difficulteData = $request->input('Difficulté opératoire')) {
                $intervention->difficulteOperatoire()->create([
                    'sur_composante' => $difficulteData['Sur'] ?? null,
                    'nature' => json_encode($difficulteData['Nature']['value'] ?? []),
                    'nature_autre' => $difficulteData['Nature']['autre'] ?? null,
                ]);
            }

            DB::commit();

            return response()->json([
                'message' => 'Formulaire enregistré avec succès !',
                'intervention_id' => $intervention->id
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erreur store formulaire: ' . $e->getMessage() . ' Fichier: ' . $e->getFile() . ' Ligne: ' . $e->getLine());

            return response()->json([
                'message' => 'Une erreur est survenue lors de la création.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Affiche une intervention spécifique avec toutes ses données formatées.
     * READ (Détail)
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id L'ID de l'intervention à afficher
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        // Validation de l'UID du chirurgien
        $request->validate([
            'surgeon_firebase_uid' => 'required|string|exists:surgeons,firebase_uid'
        ]);
        
        // Récupération du chirurgien authentifié
        $surgeon = Surgeon::where('firebase_uid', $request->query('surgeon_firebase_uid'))->first();

        // Récupération sécurisée de l'intervention avec toutes ses relations
        $intervention = Intervention::where('surgeon_id', $surgeon->id)
            ->with([
                'patient', 'antecedentsTerrains', 'donneesCliniques', 'donneesBiologiques',
                'imageriesDiagnostics', 'implants', 'donneesAnesthesiques',
                'suitesOperatoires', 'revision', 'detailsTherapeutiquesImplants',
                'greffeOsseuse', 'systemeNavigation', 'analgesie', 
                'prophylaxieAntibiotique', 'difficulteOperatoire'
            ])
            ->findOrFail($id);

        // Construction de la réponse JSON structurée
        $response = [
            'id' => $intervention->id,
            'formType' => $intervention->type_prothese,
            'IDENTITE DU PATIENT' => $this->buildPatientIdentitySection($intervention),
            'TERRAIN (S) - ANTECEDENTS' => $this->buildAntecedentsSection($intervention),
            'TERRAIN (S) - ANTECEDANTS' => $this->buildAntecedentsSection($intervention), // Alias pour le genou
            'DONNEES CLINIQUES' => $this->buildClinicalDataSection($intervention),
            'DONNEES PARACLINIQUES' => $this->buildParaclinicalDataSection($intervention),
            'DONNEES THERAPEUTIQUES' => $this->buildTherapeuticDataSection($intervention),
            'Chirurgien' => $this->buildSurgeonSection($intervention),
            'Anesthésie' => $this->buildAnesthesiaSection($intervention),
            'Analgésie par infiltration locale ou loco-régionale' => $this->buildAnalgesiaSection($intervention),
            'Prophylaxie antibiotique' => $this->buildAntibioticProphylaxisSection($intervention),
            'Greffe osseuse' => $this->buildBoneGraftSection($intervention),
            'Système de navigation' => $this->buildNavigationSystemSection($intervention),
            'Difficulté opératoire' => $this->buildOperativeDifficultySection($intervention),
            'implants_traçabilité' => $this->buildImplantsTraceabilitySection($intervention),
            'Suites opératoires et Résultats' => $this->buildPostOperativeSection($intervention),
            'En cas de révision' => $this->buildRevisionSection($intervention),
        ];

        // Ajout des sections spécifiques selon le type d'intervention
        if ($intervention->type_prothese === 'hanche') {
            $response['PTH (Arthroplastie totale) 1ère intention'] = $this->buildHipSpecificSection($intervention);
        } elseif ($intervention->type_prothese === 'genou') {
            $response['DONNEES THERAPEUTIQUES']['Type d\'arthroplastie'] = $this->buildKneeArthroplastySection($intervention);
            $response['Implants utilisés'] = $this->buildKneeImplantsSection($intervention);
        }
        
        // Fusion avec les données additionnelles
        $additionalData = $this->decodeJsonField($intervention->donnees_formulaire_autres);
        $finalResponse = array_merge($response, $additionalData);

        return response()->json($finalResponse);
    }

    /**
     * Helper pour décoder un champ JSON
     */
    private function decodeJsonField($value, $default = [])
    {
        if (is_string($value)) {
            $decoded = json_decode($value, true);
            return $decoded !== null ? $decoded : $default;
        }
        return is_array($value) ? $value : $default;
    }

    /**
     * Helper pour accéder aux propriétés d'une relation nullable
     */
    private function safeGetProperty($relation, $property, $default = null)
    {
        return $relation ? $relation->{$property} : $default;
    }

    /**
     * Construction de la section identité du patient
     */
    private function buildPatientIdentitySection($intervention)
    {
        $patient = $intervention->patient;
        $dateNaissance = null;
        
        if ($patient->date_naissance) {
            $dateNaissance = is_string($patient->date_naissance) 
                ? $patient->date_naissance 
                : $patient->date_naissance->format('Y-m-d');
        }

        return [
            'Nom' => $patient->nom,
            'Prénom(s)' => $patient->prenom,
            'Date de naissance (JJ-MM-AAAA)' => $dateNaissance,
            'Sexe' => $patient->sexe,
            'Profession' => [
                'value' => $patient->profession,
                'autre' => $patient->profession_autre,
            ],
            'Adresse du patient' => $patient->adresse,
            'Téléphone' => $patient->telephone,
            'Hôpital / numéro d\'établissement' => $intervention->hopital_etablissement,
            'Numéro Identifiant' => $patient->numero_identifiant,
        ];
    }

    /**
     * Construction de la section antécédents
     */
    private function buildAntecedentsSection($intervention)
    {
        $antecedents = $intervention->antecedentsTerrains;
        
        return [
            'Antécédents médicaux' => $this->decodeJsonField(
                $this->safeGetProperty($antecedents, 'antecedents_medicaux'), []
            ),
            'Sérologie rétroviral' => $this->safeGetProperty($antecedents, 'serologie_retroviral'),
            'Traumatisme' => $this->safeGetProperty($antecedents, 'traumatisme') ? 'Oui' : 'Non',
            'Corticothérapie' => $this->safeGetProperty($antecedents, 'corticotherapie') ? 'Oui' : 'Non',
            'Intervention chirurgicale sur cette hanche' => [
                'value' => $this->safeGetProperty($antecedents, 'intervention_anterieure'),
                'autre' => $this->safeGetProperty($antecedents, 'intervention_anterieure_autre'),
            ],
            'Intervention chirurgicale sur ce genou' => [
                'value' => $this->safeGetProperty($antecedents, 'intervention_anterieure'),
                'autre' => $this->safeGetProperty($antecedents, 'intervention_anterieure_autre'),
            ],
        ];
    }

    /**
     * Construction de la section données cliniques
     */
    private function buildClinicalDataSection($intervention)
    {
        $cliniques = $intervention->donneesCliniques;
        
        return [
            'Poids (kg)' => $this->safeGetProperty($cliniques, 'poids_kg'),
            'Taille (cm)' => $this->safeGetProperty($cliniques, 'taille_cm'),
            'IMC (kg/m²)' => $this->safeGetProperty($cliniques, 'imc'),
            'Côté' => $this->safeGetProperty($cliniques, 'cote_opere'),
            'Boiterie' => $this->safeGetProperty($cliniques, 'boiterie') ? 'Oui' : 'Non',
            'Impotence fonctionnelle' => [
                'value' => $this->safeGetProperty($cliniques, 'impotence_fonctionnelle') ? 'Oui' : 'Non',
                'conditional_value' => $this->safeGetProperty($cliniques, 'impotence_fonctionnelle_precision'),
            ],
            'PMA' => $this->safeGetProperty($cliniques, 'pma_pre_operatoire'),
            'Score de Parker' => $this->safeGetProperty($cliniques, 'score_parker'),
            'ILMI' => [
                'value' => $this->safeGetProperty($cliniques, 'ilmi') ? 'Oui' : 'Non',
                'autre' => $this->safeGetProperty($cliniques, 'ilmi_cm'),
            ],
        ];
    }

    /**
     * Construction de la section données paracliniques
     */
    private function buildParaclinicalDataSection($intervention)
    {
        $biologiques = $intervention->donneesBiologiques;
        $imageries = $intervention->imageriesDiagnostics;
        
        return [
            'Biologie' => $biologiques ? [
                'Globules Blancs (10³/mm³)' => $biologiques->globules_blancs,
                'Neutrophiles' => $biologiques->neutrophiles,
                'Hémoglobine (g/dl)' => $biologiques->hemoglobine,
                'Plaquettes (10³/mm³)' => $biologiques->plaquettes,
                'Groupe Sanguin' => $biologiques->groupe_sanguin,
                'Rhésus' => $biologiques->rhesus,
                'Créatinine (mmol/L)' => $biologiques->creatinine,
                'Glycémie (mmol/L)' => $biologiques->glycemie,
                'CRP (g/L)' => $biologiques->crp,
                'Electrophorèse de l\'hémoglobine' => $biologiques->electrophorese_hemoglobine,
                'Procalcitonine' => $biologiques->procalcitonine,
                'TP/TCK' => $biologiques->tp_tck,
                'INR' => $biologiques->inr,
                'Autre (préciser)' => $biologiques->autre_biologie,
            ] : null,
            'Imagerie (Diagnostic)' => $imageries ? [
                'Techniques' => $this->decodeJsonField($imageries->techniques, []),
                'Résultats (cocher le principal)' => [
                    'value' => $imageries->resultat_principal,
                    'autre' => $imageries->resultat_principal_autre,
                ],
                'Diagnostic (cocher le principal)' => [
                    'value' => $imageries->resultat_principal,
                    'autre' => $imageries->resultat_principal_autre,
                ],
            ] : null,
        ];
    }

    /**
     * Construction de la section données thérapeutiques
     */
    private function buildTherapeuticDataSection($intervention)
    {
        return [
            'Chirurgicales' => [
                'Date intervention' => $intervention->date_intervention,
                'Heure début' => $intervention->heure_debut,
                'Fin' => $intervention->heure_fin,
                'Voie d\'abord de la hanche' => [
                    'value' => $intervention->voie_dabord,
                    'autre' => $intervention->voie_dabord_autre,
                ],
                'Voie d\'abord' => $intervention->voie_dabord,
            ],
        ];
    }

    /**
     * Construction de la section chirurgien
     */
    private function buildSurgeonSection($intervention)
    {
        return [
            'Identifiant/code' => $intervention->chirurgien_identifiant,
        ];
    }

    /**
     * Construction de la section anesthésie
     */
    private function buildAnesthesiaSection($intervention)
    {
        $anesthesie = $intervention->donneesAnesthesiques;
        
        return $anesthesie ? [
            'Type' => [
                'value' => $anesthesie->type_anesthesie,
                'autre' => $anesthesie->type_anesthesie_autre,
            ],
            'Classification ASA' => $anesthesie->classification_asa,
            'Complications per-anesthésiques' => [
                'value' => $anesthesie->complications_per_anesthesiques ? 'Oui' : 'Non',
                'autre' => $anesthesie->complications_per_anesthesiques_details,
            ],
        ] : null;
    }

    /**
     * Construction de la section analgésie
     */
    private function buildAnalgesiaSection($intervention)
    {
        $analgesie = $intervention->analgesie;
        
        return $analgesie ? [
            'Infiltration' => $analgesie->infiltration ? 'Oui' : 'Non',
            'Cathéter laissé en place' => $analgesie->catheter_en_place ? 'Oui' : 'Non',
            'Produit' => $analgesie->produit,
        ] : null;
    }

    /**
     * Construction de la section prophylaxie antibiotique
     */
    private function buildAntibioticProphylaxisSection($intervention)
    {
        $prophylaxie = $intervention->prophylaxieAntibiotique;
        
        return $prophylaxie ? [
            'Utilisée' => 'Oui',
            'Si oui préciser antibiotique' => $prophylaxie->antibiotique,
            'Dose /jour' => $prophylaxie->dose_par_jour,
            'Débuté en per-opératoire' => $prophylaxie->debute_per_operatoire ? 'Oui' : 'Non',
            'Durée' => $prophylaxie->duree,
        ] : ['Utilisée' => 'Non'];
    }

    /**
     * Construction de la section greffe osseuse
     */
    private function buildBoneGraftSection($intervention)
    {
        $greffe = $intervention->greffeOsseuse;
        
        return $greffe ? [
            'Transplantation osseuse' => 'Oui',
            'Type' => $greffe->type,
            'Localisation' => is_string($greffe->localisation) 
                ? ($this->decodeJsonField($greffe->localisation) ?: $greffe->localisation)
                : $greffe->localisation,
        ] : ['Transplantation osseuse' => 'Non'];
    }

    /**
     * Construction de la section système de navigation
     */
    private function buildNavigationSystemSection($intervention)
    {
        $navigation = $intervention->systemeNavigation;
        
        return $navigation ? [
            'Utilisation' => 'Oui',
            'Système utilisé' => $navigation->systeme_utilise,
            'Numéro de série' => $navigation->numero_serie,
        ] : ['Utilisation' => 'Non'];
    }

    /**
     * Construction de la section difficulté opératoire
     */
    private function buildOperativeDifficultySection($intervention)
    {
        $difficulte = $intervention->difficulteOperatoire;
        
        return $difficulte ? [
            'Sur' => $difficulte->sur_composante,
            'Nature' => [
                'value' => $this->decodeJsonField($difficulte->nature, []),
                'autre' => $difficulte->nature_autre,
            ],
        ] : null;
    }

    /**
     * Construction de la section traçabilité des implants
     */
    private function buildImplantsTraceabilitySection($intervention)
    {
        return $intervention->implants->map(function($implant) {
            return [
                'type_composant' => $implant->type_composant,
                'societe' => $implant->societe,
                'marque_modele' => $implant->marque_modele,
                'reference' => $implant->reference,
                'numero_lot' => $implant->numero_lot,
                'udi_di_pi' => $implant->udi_di_pi,
                'date_fabrication' => $implant->date_fabrication,
                'pays_fabrication' => $implant->pays_fabrication,
            ];
        })->toArray();
    }

    /**
     * Construction de la section suites opératoires
     */
    private function buildPostOperativeSection($intervention)
    {
        $suites = $intervention->suitesOperatoires;
        
        return $suites ? [
            'Délai de mise en charge (jours)' => $suites->delai_mise_en_charge_jours,
            'Délai d\'appui total (mois)' => $suites->delai_appui_total_mois,
            'Durée de séjour hospitalier' => $suites->duree_sejour_hospitalier,
            'Complications per opératoire' => $this->decodeJsonField($suites->complications_per_operatoire, []),
            'Suites opératoires immédiates' => $suites->suites_immediates,
            'Complications immédiates' => $this->decodeJsonField($suites->complications_immediates, []),
            'Complications secondaires' => $this->decodeJsonField($suites->complications_secondaires, []),
            'Complications tardives' => $this->decodeJsonField($suites->complications_tardives, []),
            'Prise en charge des frais de soins' => [
                'value' => $suites->prise_en_charge_frais,
                'autre' => $suites->prise_en_charge_frais_autre,
            ],
            'Dernier recul (mois)' => $suites->dernier_recul_mois,
            'Appréciation du patient lui-même : EVA' => $suites->eva_patient,
            'Evaluation fonctionnelle : PMA' => $suites->pma_post_operatoire,
        ] : null;
    }

    /**
     * Construction de la section révision
     */
    private function buildRevisionSection($intervention)
    {
        $revision = $intervention->revision;
        
        return $revision ? [
            'Type de révision' => [
                'Type' => $this->decodeJsonField($revision->types_revision, []),
            ],
            'Motif de la révision' => [
                'Motif' => $this->decodeJsonField($revision->motifs_revision, []),
            ],
        ] : null;
    }

    /**
     * Construction de la section spécifique à la hanche
     */
    private function buildHipSpecificSection($intervention)
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        
        return $details ? [
            'Couple de frottement' => [$details->couple_frottement],
            'Composant acétabulaire' => [
                'Type de cupule' => $details->type_cupule,
            ],
            'Tige fémorale' => [
                'Collerette' => $details->tige_femoral_collerette,
                'Type de fixation' => $details->tige_femoral_fixation,
            ],
            'Resurfaçage' => [$details->resurfacage ? 'Oui' : 'Non'],
        ] : null;
    }

    /**
     * Construction de la section type d'arthroplastie pour le genou
     */
    private function buildKneeArthroplastySection($intervention)
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        
        return $details ? [
            'Type' => $details->type_arthroplastie,
            'Si prothèse primaire' => [
                'value' => $details->type_prothese_primaire,
            ],
        ] : null;
    }

    /**
     * Construction de la section implants utilisés pour le genou
     */
    private function buildKneeImplantsSection($intervention)
    {
        $details = $intervention->detailsTherapeutiquesImplants;
        
        return $details ? [
            'Composant fémoral' => [
                'Fixation' => $details->fixation_femoral,
            ],
            'Composant tibial (plateau)' => [
                'Fixation' => $details->fixation_tibial,
            ],
            'Insert (polyéthylène)' => [
                'Mobilité' => $details->mobilite_insert,
                'Design' => $details->design_insert,
            ],
            'Composant patellaire' => [
                'Patella posée' => $details->patella_posee ? 'Oui' : 'Non',
            ],
        ] : null;
    }

        /**
     * Met à jour une intervention existante et toutes ses données associées.
     * C'est la méthode UPDATE.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id L'ID de l'intervention à mettre à jour
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        // On valide que l'UID du chirurgien est bien présent dans le corps de la requête.
        $request->validate(['surgeon_firebase_uid' => 'required|string|exists:surgeons,firebase_uid']);
        
        // On récupère le chirurgien qui fait la demande.
        $surgeon = Surgeon::where('firebase_uid', $request->input('surgeon_firebase_uid'))->first();

        // REQUÊTE SÉCURISÉE : On ne cherche l'intervention à mettre à jour
        // que parmi celles qui appartiennent au chirurgien authentifié.
        $intervention = Intervention::where('surgeon_id', $surgeon->id)->findOrFail($id);

        // La transaction garantit que la mise à jour est atomique : soit tout réussit, soit tout est annulé.
        DB::beginTransaction();
        try {
            // == 1. Mise à jour du Patient ==
            $patientData = $request->input('IDENTITE DU PATIENT');
            $intervention->patient()->update([
                'nom' => $patientData['Nom'],
                'prenom' => $patientData['Prénom(s)'],
                'date_naissance' => $patientData['Date de naissance (JJ-MM-AAAA)'],
                'sexe' => $patientData['Sexe'],
                'profession' => $patientData['Profession']['value'] ?? null,
                'profession_autre' => $patientData['Profession']['autre'] ?? null,
                'adresse' => $patientData['Adresse du patient'],
                'telephone' => $patientData['Téléphone'],
            ]);

            // == 2. Mise à jour de l'Intervention (pivot) ==
            $intervention->update([
                'type_prothese' => $request->input('formType'),
                'hopital_etablissement' => $patientData['Hôpital / numéro d\'établissement'],
                'numero_identifiant_patient' => $patientData['Numéro Identifiant'],
                'date_intervention' => $request->input('DONNEES THERAPEUTIQUES.Chirurgicales.Date intervention'),
                'heure_debut' => $request->input('DONNEES THERAPEUTIQUES.Chirurgicales.Heure début'),
                'heure_fin' => $request->input('DONNEES THERAPEUTIQUES.Chirurgicales.Fin'),
                'chirurgien_identifiant' => $request->input('Chirurgien.Identifiant/code'),
                'donnees_formulaire_autres' => \Illuminate\Support\Arr::except($request->all(), [
                    'surgeon_firebase_uid','surgeon_name','formType','IDENTITE DU PATIENT',
                    'TERRAIN (S) - ANTECEDENTS','DONNEES CLINIQUES','DONNEES PARACLINIQUES',
                    'Chirurgien','Anesthésie','implants_traçabilité','Suites opératoires et Résultats',
                    'En cas de révision'
                ])
            ]);

            // == 3. Mise à jour des relations "hasOne" avec updateOrCreate ==
            // updateOrCreate met à jour la ligne si elle existe, ou la crée si elle n'existe pas. C'est idéal.
            $antecedentsData = $request->input('TERRAIN (S) - ANTECEDENTS');
            $intervention->antecedentsTerrains()->updateOrCreate(['intervention_id' => $intervention->id], [
                'antecedents_medicaux' => $antecedentsData['Antécédents médicaux'],
                'serologie_retroviral' => $antecedentsData['Sérologie rétroviral'],
                'traumatisme' => ($antecedentsData['Traumatisme'] ?? 'Non') === 'Oui',
                'corticotherapie' => ($antecedentsData['Corticothérapie'] ?? 'Non') === 'Oui',
                'intervention_anterieure' => $antecedentsData['Intervention chirurgicale sur cette hanche']['value'] ?? $antecedentsData['Intervention chirurgicale sur ce genou']['value'] ?? null,
                'intervention_anterieure_autre' => $antecedentsData['Intervention chirurgicale sur cette hanche']['autre'] ?? $antecedentsData['Intervention chirurgicale sur ce genou']['autre'] ?? null,
            ]);

            $cliniquesData = $request->input('DONNEES CLINIQUES');
            $intervention->donneesCliniques()->updateOrCreate(['intervention_id' => $intervention->id], $cliniquesData);

            $bioData = $request->input('DONNEES PARACLINIQUES.Biologie');
            $intervention->donneesBiologiques()->updateOrCreate(['intervention_id' => $intervention->id], (array) $bioData);

            $imagerieData = $request->input('DONNEES PARACLINIQUES.Imagerie (Diagnostic)');
            $intervention->imageriesDiagnostics()->updateOrCreate(['intervention_id' => $intervention->id], [
                'techniques' => $imagerieData['Techniques'],
                'resultat_principal' => $imagerieData['Résultats (cocher le principal)']['value'] ?? $imagerieData['Diagnostic (cocher le principal)']['value'] ?? null,
                'resultat_principal_autre' => $imagerieData['Résultats (cocher le principal)']['autre'] ?? $imagerieData['Diagnostic (cocher le principal)']['autre'] ?? null,
            ]);

            $anesthesieData = $request->input('Anesthésie');
            $intervention->donneesAnesthesiques()->updateOrCreate(['intervention_id' => $intervention->id], [
                'type_anesthesie' => $anesthesieData['Type']['value'] ?? null,
                'type_anesthesie_autre' => $anesthesieData['Type']['autre'] ?? null,
                'classification_asa' => $anesthesieData['Classification ASA'],
                'complications_per_anesthesiques' => ($anesthesieData['Complications per-anesthésiques']['value'] ?? 'Non') === 'Oui',
                'complications_per_anesthesiques_details' => $anesthesieData['Complications per-anesthésiques']['autre'] ?? null,
            ]);

            $suitesData = $request->input('Suites opératoires et Résultats');
            if ($suitesData) {
                $intervention->suitesOperatoires()->updateOrCreate(['intervention_id' => $intervention->id], $suitesData);
            }

            $revisionData = $request->input('En cas de révision');
            if ($revisionData) {
                $intervention->revision()->updateOrCreate(['intervention_id' => $intervention->id], [
                    'est_une_revision' => true,
                    'types_revision' => $revisionData['Type de révision']['Type'],
                    'motifs_revision' => $revisionData['Motif de la révision']['Motif'],
                ]);
            }

            // == 4. Mise à jour des Implants (hasMany) : Supprimer et recréer ==
            $intervention->implants()->delete(); // On supprime les anciens
            $implantsData = $request->input('implants_traçabilité', []);
            foreach ($implantsData as $implant) {
                $intervention->implants()->create($implant); // On insère les nouveaux
            }

            DB::commit();

            return response()->json(['message' => 'Formulaire mis à jour avec succès !', 'intervention_id' => $intervention->id]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erreur update formulaire: ' . $e->getMessage() . ' Ligne: ' . $e->getLine());
            return response()->json(['message' => 'Une erreur est survenue lors de la mise à jour.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Supprime une intervention et toutes ses données associées,
     * après avoir vérifié que l'action est autorisée.
     * C'est la méthode DELETE.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id L'ID de l'intervention à supprimer
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request, $id)
    {
        // On valide que l'UID du chirurgien a bien été fourni dans le corps de la requête.
        $request->validate(['surgeon_firebase_uid' => 'required|string|exists:surgeons,firebase_uid']);

        // On récupère le chirurgien qui effectue la demande.
        $surgeon = Surgeon::where('firebase_uid', $request->input('surgeon_firebase_uid'))->first();

        // REQUÊTE SÉCURISÉE : On cherche l'intervention à supprimer
        // uniquement parmi celles qui appartiennent au chirurgien authentifié.
        $intervention = Intervention::where('surgeon_id', $surgeon->id)->findOrFail($id);
        // Si l'intervention n'est pas trouvée ou n'appartient pas au chirurgien,
        // findOrFail renverra automatiquement une erreur 404 (Not Found), empêchant toute action non autorisée.

        // ACTION DE SUPPRESSION CORRECTE : On supprime l'intervention.
        // Grâce à `onDelete('cascade')` dans nos migrations, toutes les données dans les tables
        // 'antecedents_terrains', 'donnees_cliniques', 'implants', etc. liées à cette intervention
        // seront automatiquement supprimées par la base de données.
        // Le patient, lui, ne sera PAS supprimé.
        $intervention->delete();

        // On retourne une réponse HTTP 204 No Content, qui est la norme pour
        // une suppression réussie qui n'a pas besoin de renvoyer de contenu.
        return response()->json(null, 204);
    }
}